function OptResult=optimizeSystem(MySystem,params,Ubounds,Lbounds,InitialG,targetvalues,objfunction,varargin)
% This function optimizes a Thermoacoustic system using MATLAB's surrogate
% optimization algortithm.
%Standard format for running is
%OptResult=optimizeSystem(MySystem,params,Ubounds,Lbounds,InitialG,targetvalues,objfunction)

% input parameters: 
% MySystem-     The system to be optimized, system must be converged
% params-       cell array of system parameters to be otpimized, given in the
%               standard PCTAS format (e.g. {{'Frequency'},{'Duct_1' 'Length'}})
% Ubounds-      upper bounds for parameters, must be a vector of the same length as params
% Lbounds-      Lower bounds for parameters, must be a vector of the same length as params
% InitialG-     current converged value of guesses
% targetvalues- values for system targets
% objfunction - a handle to an objective function to be MINIMISED. The 
%               function must accept the  thermoacoustic system as an input
%               and return as an output a function value to be minimized.
%               e.g if we wish to improve the heat input Q we could choose
%               1/Q or -Q as a minimization factor

%output parameter
% OptResult- a structure including the optimized value X, the value of the
%            objective function at the optimum point fval, the exit flag
%            from the optimization process and additional data in output.

% Running with check-points:
% you can run the system and save the results to a check-point file, by
% adding it to the input parameters:
% OptResult=optimizeSystem(MySystem,...,objfunction,CP_Path)
% CP_Path is a path (relative to the working directory) for the checkpoint
% file. if the file does not exist yet, it will be created and saved during
% the optimization process. If the file does exist, the optimization
% process wil continue from check-point.

%collecting data bases
%during the run a data base of points (of X) and their corresponding 
% converged values is collected. If you want to save and/or load a database
%file, you can add a path to it:
% 
% OptResult=optimizeSystem(MySystem,...,objfunction,CP_Path,Databsepath)
% OptResult=optimizeSystem(MySystem,...,objfunction,[],Databsepath)


%% check that system converges with Initial Guess
try
    [~,~,~,info]=MySystem.Run_System_GT(InitialG,targetvalues);
catch
    info=2;
end
if info~=1
    error('System does not converge with initial guess')
end
%% colect initial values of optimization variable
InitialParams=nan(size(params));
for i=1:length(params)
    InitialParams(i)=MySystem.Evaluate_Property(params{i});
end


%% construct InitialPoints 
%DB indicates if a database exists or not
if isempty(varargin)||length(varargin)<2||isempty(varargin{2})
   DB=false;  
   DBpath=[];
elseif isfile(varargin{2})
    DB=true;
    DBpath=varargin{2};
else
   DB=true;
   DBpath=varargin{2};
   S=struct('Xss',InitialParams,'guessvalues',InitialG,'RunTimes',[]);
   save(DBpath,'-struct','S')
end
%% define and run optimization algorithm
fun=@(x)(Run_objective_function(x,MySystem,params,targetvalues...
            ,InitialParams,InitialG,objfunction,DB,DBpath));

if isempty(varargin)||isempty(varargin{1})
    options = optimoptions('surrogateopt');
    [x,fval,exitflag,output] = surrogateopt(fun,Lbounds,Ubounds,options);
elseif isfile(varargin{1}) %if there exists a checkpointfile
    [x,fval,exitflag,output] = surrogateopt(varargin{1});
else
    options = optimoptions('surrogateopt','CheckpointFile',varargin{1});
    [x,fval,exitflag,output] = surrogateopt(fun,Lbounds,Ubounds,options);
end

%% assign output
OptResult.x=x;
OptResult.fval=fval;
OptResult.exitflag=exitflag;
OptResult.output=output;



%% end of main function
%% Objective function
% this function calculates the objective function objfunction for
% a given input x. It first changes the system smoothly to the given input
% x using 5 steps, if the change fails it does the same with a smart change
% using 100 steps,
% and if this fails as well it tries a smart change using  100 steps. If
% all fails it will simply return 100, indicating poor performance
    function y=Run_objective_function(x,MySystem,params,targetvalues...
            ,InitialParams,InitialG,objfunction,DB,DBpath)
        tic %start timing runtime
        if DB
            load(DBpath,"Xss","guessvalues","RunTimes")
        else
            Xss=InitialParams;
            guessvalues=InitialG;
            RunTimes=[];
        end
        %try to set the system to the nearest converged value
        disp(['Running the objective function with x=',num2str(x)])
        disp('looking for nearest neighbour in database')
        try
            difMat=abs(Xss-x)./abs(Xss);
            norms = sqrt(sum(difMat.^2, 2));
            [~, minIndex] = min(norms);
            disp(['nearest neighbour is x=',num2str(Xss(minIndex,:))...
                '!, running at nearest neighbour'])
            success=setsystem(MySystem,params,Xss(minIndex,:),guessvalues(minIndex,:),targetvalues);
        catch
            success=0;
        end

        if ~success
            disp('something went wrong with nearst neighbour. changing from initial value');
            setsystem(MySystem,params,InitialParams,InitialG,targetvalues);
        end
        
        % change the function to calculated point using ChangeSmoothMulti
        disp('trying changing with 5 steps')
        try
            result=ChangeSmoothMulti(MySystem,params,x,5,'NoQuestions','DispIterations','CollectData');
        catch
            result.Converged=0;
            setsystem(MySystem,params,InitialParams,InitialG,targetvalues);
        end
        if ~result.Converged
            disp('did not converge. trying changing with 10 steps (Smart)')
            try
                result=ChangeSmoothMulti(MySystem,params,x,10,'NoQuestionsSmart','DispIterations','CollectData');
            catch
                result.Converged=0;
                setsystem(MySystem,params,InitialParams,InitialG,targetvalues);
            end
            if ~result.Converged
                disp('did not converge. trying changing with 100 steps (Smart)')
                try
                    result=ChangeSmoothMulti(MySystem,params,x,100,'NoQuestionsSmart','DispIterations','CollectData');
                catch
                    result.Converged=0;
                    setsystem(MySystem,params,InitialParams,InitialG,targetvalues);
                end

            end
        end
        %check if result is converged. if converged, add guesses and Xss to
        %guess database. if not, return 100 and leave.
  
        if result.Converged
            y=objfunction(MySystem);
            guessvalues=[guessvalues;result.Converged_Value];
            Xss=[Xss;result.AllVals];
        else
            disp('did not converge. returning y=100')
            y=100;
            setsystem(MySystem,params,InitialParams,InitialG,targetvalues);
        end
        RunTimes=[RunTimes;toc];
        if DB
            save(DBpath,"Xss","guessvalues","RunTimes")
        end

    end
%% assisting functions
%set system to a specific value once the converged guess is known
    function success=setsystem(MySystem,params,WantedParams,knownG,targetvalues) 
        for j=1:length(params)
            MySystem.Change_Property(params{j},WantedParams(j))
        end
        try
        [~,Fnorm, ~,  Info]=MySystem.Run_System_GT(knownG,targetvalues,'Smart');
        catch
            Info=2;
        end
        success=(Info==1&&Fnorm<1e-5);
    end
end