classdef TA_Rectangular_Stack <TA_Stack
    %describes a rectangular pore stack.
    %% independent properties
    properties (SetObservable=true)
        Y1(1,1) double{mustBeNonnegative} %half length of one pore side [m]
        Y2(1,1) double{mustBeNonnegative} %half length of other side    [m]
        Lp(1,1) double{mustBeNonnegative} %half thickness of solid      [m]
    end 

    properties (Hidden) % independent connected properties
    % parameters which keep values and connects to the other parameters:
        Diameter_val (1,1) double{mustBeNonnegative} 
        Area_val=0.1 
        A_Solid_val=0
        Gas_Area_Ratio_val (1,1) double{mustBeGreaterThan(Gas_Area_Ratio_val,0)...
            ,mustBeLessThan(Gas_Area_Ratio_val,1)} = 0.1 
    end

    %% dependent geometrical properties
    properties (SetObservable=true,Dependent) % dependent properties
        Diameter(1,1) double{mustBeNonnegative} % stack diameter [m]
        Area           % stack area [m^2]         [m]?
        A_Solid           % Area covered by solid    [m]?
        Gas_Area_Ratio...   % percentege of area covered by gas
            (1,1) double{mustBeGreaterThan(Gas_Area_Ratio,0)...
            ,mustBeLessThan(Gas_Area_Ratio,1)}
    end
    
    methods
        %% constructor
        function obj = TA_Rectangular_Stack(name,system,length,D,y1,y2,phi,solid,varargin)
            obj@TA_Stack(name,system,length,solid,varargin{:})
            obj.Diameter_val=D;
            obj.Y1=y1;
            obj.Y2=y2;
            obj.Gas_Area_Ratio_val=phi;
            obj.Lp=(y1*y2*(1/phi-1))/(y1+y2);
            obj.Area=pi*D^2/4;
            obj.A_Solid_val=pi*D^2/4*(1-obj.Gas_Area_Ratio_val);
        end
        %% set method for diameter
        function set.Diameter(obj,value)
            obj.Diameter_val=value;
            obj.Area_val=pi*value^2/4;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*obj.Area_val;
        end
        %% get method for diameter
        function value=get.Diameter(obj)
            value=obj.Diameter_val;
        end

        %% set method for area
        function set.Area(obj,value)
            obj.Area_val=value;
            obj.Diameter_val=(value/pi*4)^0.5;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*value;
        end
        %% get method for area
        function value=get.Area(obj)
             value=obj.Area_val;
        end

        %% set method for A_solid
        function  set.A_Solid(obj,value)
            if value>=obj.Area_val
                error('solid area must be smaller than total area')
            end
            obj.A_Solid_val=value;
            obj.Gas_Area_Ratio_val=1-value/obj.Area_val;
        end
        %% get method for A_Solid
        function value=get.A_Solid(obj)
             value=obj.A_Solid_val;
        end

        %% set method for Gas_Area
        function  set.Gas_Area_Ratio(obj,value)
            if value>=1
                error('solid area must be smaller than total area')
            end
            obj.Gas_Area_Ratio_val=value;
            obj.A_Solid_val=(1-value)*obj.Area_val;
        end
        %% get method for Gas_Area_Ratio
        function value=get.Gas_Area_Ratio(obj)
             value=obj.Gas_Area_Ratio_val;
        end
        
        %% geometrical functions
        %this function  assists in averaging a property based
        %on the thermal penetration depth and geometryx
        %source:delta E.C version 6.4b2.7 user's guide, equations
        %10.59-10.60
        
        function  f=f_function(obj,delta)
            a=obj.Y1;
            b=obj.Y2;
            %initialize the sum to be added to f
            sum=0;
            for m=1:2:31
                for n=1:2:31
                    YYY=1-1i*pi^2*delta^2/(8*a^2*b^2)*(b^2*m^2+a^2*n^2);
                    sum=sum+1/(m^2*n^2*YYY);    
                end
                f=1-sum*64/pi^4;
            end
        end
        function epsilon=eps_function(obj,deltak,deltas,fk)
            a=obj.Y1;
            b=obj.Y2;
            l=obj.Lp;
            epsilon=fk*(1+1i)*a*b/(deltak*(a+b))/tanh((1+1i)*l/deltas);
            
        end
    end
end

