classdef TA_Plate_Stack <TA_Stack
    %Describes a parralel plate stack

    properties (SetObservable=true) % independent properties
        Y0 (1,1) double{mustBeNonnegative}    % half the plate spacing  [m]
        Lp (1,1) double{mustBeNonnegative}    % half thickness of solid [m]
    end

    properties (Hidden) % independent properties
    % parameters which keep values and connects to the other parameters:
        Diameter_val 
        Area_val
        A_Solid_val
        Gas_Area_Ratio_val (1,1) double{mustBeGreaterThan(Gas_Area_Ratio_val,0)...
            ,mustBeLessThan(Gas_Area_Ratio_val,1)} = 0.1
    end
        
    properties (SetObservable=true,Dependent) % dependent properties
        Diameter (1,1) double{mustBeNonnegative}  % stack diameter [m]
        Area           % stack area [m^2]         [m]?
        A_Solid       % Area covered by solid    [m]?
        Gas_Area_Ratio...   % percentege of area covered by gas
            (1,1) double{mustBeGreaterThan(Gas_Area_Ratio,0)...
            ,mustBeLessThan(Gas_Area_Ratio,1)}
     end
 
    methods
        %% constructor
        function obj = TA_Plate_Stack(name,system,length,D,y0,phi,solid,varargin)
            obj@TA_Stack(name,system,length,solid,varargin{:})
            obj.Diameter_val=D;
            obj.Area_val=pi*D^2/4;
            obj.Gas_Area_Ratio_val=phi;
            obj.A_Solid_val=obj.Area_val*(1-obj.Gas_Area_Ratio_val);
            obj.Y0=y0;
            obj.Lp=y0*(1/phi-1);
        end
        %% set method for diameter
        function set.Diameter(obj,value)
            obj.Diameter_val=value;
            obj.Area_val=pi*value^2/4;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*obj.Area_val;
        end
        %% get method for diameter
        function value=get.Diameter(obj)
            value=obj.Diameter_val;
        end

        %% set method for area
        function set.Area(obj,value)
            obj.Area_val=value;
            obj.Diameter_val=(value/pi*4)^0.5;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*value;   
        end
        %% get method for area
        function value=get.Area(obj)
             value=obj.Area_val;
        end

        %% set method for A_solid
        function  set.A_Solid(obj,value)
            if value>=obj.Area_val
                error('solid area must be smaller than total area')
            end
            obj.A_Solid_val=value;
            obj.Gas_Area_Ratio_val=(obj.Area_val-value)/obj.Area_val;
        end
        %% get method for A_Solid
        function value=get.A_Solid(obj)
             value=obj.A_Solid_val;
        end

        %% set method for Gas_Area ratio
        function  set.Gas_Area_Ratio(obj,value)
            obj.Gas_Area_Ratio_val=value;
            obj.A_Solid_val=(1-value)*obj.Area_val;
        end
        %% get method for Gas_Area_Ratio
        function value=get.Gas_Area_Ratio(obj)
             value=obj.Gas_Area_Ratio_val;
        end

        %% geometrical functions
        %this function  assists in averaging a property based
        %on the thermal penetration depth and geometry
        %source:delta E.C version 6.4b2.7 user's guide, equations
        %10.59-10.60
        
        function  f=f_function(obj,delta)
            rh=obj.Y0;
            f=tanh((1+1i)*rh/delta)/((1+1i)*rh/delta);
        end
        function epsilon=eps_function(obj,deltak,deltas,~)
            rh=obj.Y0;
            lp=obj.Lp;
            epsilon=tanh((1+1i)*rh/deltak)/tanh((1+1i)*lp/deltas);
        end
    end
end