classdef  TA_Component<matlab.mixin.Copyable
    %abstract class, describes an thermoacoustic component
    
    properties (Abstract)
        %% unprotected properties
        % length [m]
        Length (1,1) double{mustBeNonnegative}
        % number of points in solution
        N_Sol_Points double{mustBeInteger}
    end
    properties (SetAccess=?TA_System)
        %% protected properties
        % string with name of "mother" system of the component
        System (1,:) char
        
        % the location of the duct in the mother system
        Location double{mustBeInteger}
    end
    properties(SetAccess= 'protected',SetObservable=true)
        %% variable profile
        %these are the governing equation's parameters:oscilating pressure,
        %oscillationg velocity, mean temperature Location and some derived
        % properties.
        
        %string containing variable name
        Name           (1,:) char
        
        X              (:,1) double   %the location                   [m]
        Pressure       (:,1) double   %the oscillating pressure field [pa]
        Velocity       (:,1) double   %the oscillating velocity field [m/s]
        Temperature     ...           %temperature                    [k]
            (:,1) double{mustBeNonnegative}
        
        Acoustic_Power (:,1) double   %the acoustic power [W]
        Total_Power    (:,1) double   % the total power   [W]
        Mass_Flux      (:,1) double   % the mass flux     [kg/s]


        
    end
    properties (AbortSet=true,SetObservable=true, Dependent)
        %% parameters for calculation of mixture properties
        Mixture_Array  cell
        Inert      ... %string with inert component
            {mustBeA(Inert,["cell","string","char"])}
        Reactive   ... %string with reactive component
            {mustBeA(Reactive,["cell","string","char"])}
        Dry_Switch (1,1)  %1 for dry, 0 for wet, 'null' to be synchronized by the system
        
    end
    properties (Hidden,AbortSet=true) % independent properties
        %% parameters which keep values and connects to the other parameters:
        Inert_val     ... %string with inert component, 'null' to be synchronized by the system
            {mustBeA(Inert_val,["cell","string","char"])} = "null" 
        Reactive_val  ... %string with reactive component, 'null' to be synchronized by the system
            {mustBeA(Reactive_val,["cell","string","char"])} = "null"
        Dry_Switch_val (1,1) = "null"  %1 for dry, 0 for wet, 'null' to be synchronized by the system
        Mixture_Array_val  cell = {} % 'empty' to be synchronized by the system

    end
    properties (Hidden,SetAccess=?TA_System)
        %% internal parameters for the class to work
        System_H  TA_System %handle to actual system
    end

    methods
        %% constructor
        function obj=TA_Component(name,system,varargin)
            obj.Name=name;

            if ~(isa(system,'char')&&strcmp(system,'null')) % connects to system if defined
                obj.System=system.Name;
                obj.System_H=system;
                obj.Location=size(system.Components,2)+1;
                system.addComponent(name);
                system.Components_H=[system.Components_H,{obj}];
            end

            % Adding the specific mixture properties:
            if size(varargin,2) > 1 % two inputs at least
                obj.Inert = varargin{1};
                obj.Reactive = varargin{2};
                if size(varargin,2) == 3 % three inputs
                    obj.Dry_Switch = varargin{3};
                end

            elseif ~isempty(varargin) % in case varargin is not 2-3 items or empty -> mainly error of code usage and not GUI         
                error('Wrong number of inputs.') 
            end

        end
        %% Set & get methods for inert and reactive components, as well as dryswitch
        % Set method for Inert:
        function set.Inert(obj,value)
            if ischar(value)
                value=string(value);
            elseif iscell(value)
                value{1}=string(value{1});
                value{3}=string(value{3});
            end
            obj.Inert_val=value;
            if ~isempty(obj.Reactive) && ~strcmp(obj.Reactive,'null') && ~strcmp(obj.Inert,'null')
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end
        end
        % Get method for Inert:
        function value=get.Inert(obj)
            if strcmp(obj.Inert_val, "null") % if null, returning the system's value
                value = obj.System_H.Inert;
            else
                value=obj.Inert_val;
            end
        end

        % Set method for Reactive:
        function set.Reactive(obj,value)
            if ischar(value)
                value=string(value);
            elseif iscell(value)
                value{1}=string(value{1});
                value{3}=string(value{3});
            end
            obj.Reactive_val=value;
            if ~isempty(obj.Reactive) && ~strcmp(obj.Reactive,'null') && ~strcmp(obj.Inert,'null')
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end
        end
        % Get method for Reactive:
        function value=get.Reactive(obj)
            if strcmp(obj.Reactive_val, "null") % if null, returning the system's value
                value = obj.System_H.Reactive;
            else
                value=obj.Reactive_val;
            end
        end
        
        % Set method for Mixture_Array
        function  set.Mixture_Array(obj,value)
            obj.Mixture_Array_val=value;
        end
        % Get method for Mixture_Array:
        function value=get.Mixture_Array(obj)
            if isempty(obj.Mixture_Array_val) % if empty, returning the system's value
                value = obj.System_H.Mixture_Array;
            else
                value=obj.Mixture_Array_val;
            end
        end

       % initializing saved properties when Dry_Switch is modified ~(was told to leave it for now)~
       % Set method for Dry Switch:
        function set.Dry_Switch(obj,value)
            obj.Dry_Switch_val=value;
          %  obj.saved_p=NaN;
          %  obj.saved_T=NaN;
          %  obj.saved_m_properties=NaN(16,1);
        end

        % Get method for Dry Switch:
        function value=get.Dry_Switch(obj)
            if strcmp(obj.Dry_Switch_val, "null") % if null, returning the system's value
                value = obj.System_H.Dry_Switch;
            else
                value=obj.Dry_Switch_val;
            end
        end
        
        %% rename function
        function rename(obj,value)
            oldName=obj.Name;           
            obj.Name=value;
            if ~isempty(obj.System)
                [obj.System_H.Guesses]=obj.System_H. ...
                SearchArray(obj.System_H.Guesses,oldName,value);
                [obj.System_H.Targets]=obj.System_H.SearchArray ...
                (obj.System_H.Targets,oldName,value);
                obj.System_H.Components{obj.Location}=value;
            end

            % checking if the component is associated with links and
            % changing it's name in all of them
            if ~isempty(obj.System_H.Links)
                for i = 1:length(obj.System_H.Links)
                    [obj.System_H.Links_H{i}.Linked_Properties]=obj.System_H.SearchArray(...
                        obj.System_H.Links_H{i}.Linked_Properties,oldName,value);
                end
            end
            
            
        end
        %% copy component
        function cp = Copy_Component(obj,varargin)
            %copy a component
            % Copy_Component() creates a copy of the component with the
            %                  name "copy of 'old name'"
            % Copy_Component(newName) creates a copy of the component with
            % the name specified
            
            % Shallow copy object
            cp = obj.copyElement;
            cp.System_H=TA_System.empty;
            cp.System=[];
            cp.Location=[];
            cp.Empty_Varibles;

            if isempty(varargin)
                cp.Name=['copy of ',cp.Name];
            else
                cp.Name=varargin{1};
            end
        end
        %% empty all variables (pressure, velocity etc)
        function Empty_Varibles(obj)
            obj.X=[];
            obj.Pressure=[];
            obj.Velocity=[];
            obj.Temperature=[];
            obj.Acoustic_Power=[];
            obj.Total_Power=[];
            obj.Mass_Flux=[];
        end
        %% caclulate Fs
        function [FD,Fv,Falpha] = CalculateFs(obj)
            %this function calculates the F functions FD Fv Falpha for a component
            % the F functions are defined based on Offner's notation (JFM 2019), and
            % are equal to 1-f based on the classical thermoacoustic notation
            
            % check for errors
            if isempty(obj.Pressure)
                error('Run System before collecting Fs')
            end
            
            % collect system data
            Sys= obj.System_H;
            if ~isempty(obj.Mixture_Array)
                M_Array=obj.Mixture_Array;
            else
                M_Array=Sys.Mixture_Array;
            end
            if ~isstring(obj.Dry_Switch)
            dry_switch=obj.Dry_Switch;
            else
            dry_switch=Sys.Dry_Switch;  
            end
            P_m=Sys.P_m;
            Omega=Sys.Frequency*2*pi;
            % assign fs
            
            [FD,Fv,Falpha]=deal(ones(length(obj.X),1));        %initialise variable
            
            if isa(obj,'TA_Duct')||isa(obj,'TA_Cone')  % check type of component
                % ducts and cones
                for i=1:length(obj.X)
                    %collect rh
                    if isa(obj,'TA_Duct')
                        rh=obj.Rh;                         %duct hydraulic radius
                    else
                        rh=obj.Calc_Rh(obj.X(i));%cone hydraulic radius
                    end
                    %caclulate mixture properties
                    [ nu, alpha, ~, ~, Pr, Sc] = Mixture_Properties(P_m,obj.Temperature(i),dry_switch,M_Array);
                    deltaN=(2*nu/Omega)^0.5;
                    deltaK=(2*alpha/Omega)^0.5;
                    deltaD=(Pr/Sc)^0.5*deltaK;
                    
                    %calculate f function based on different cases
                    % based on DEC 10.6-10.7
                    if rh/deltaN<12.5              %small enough for bessel
                        paramNU=(1i-1).*2*rh./deltaN; %parameter for convenicence
                        paramK=(1i-1).*2*rh./deltaK;
                        paramD=(1i-1).*2*rh./deltaD;
                        
                        fnu=(2*besselj(1,paramNU)./(besselj(0,paramNU).*paramNU));
                        fk=(2*besselj(1,paramK)./(besselj(0,paramK).*paramK));
                        fD=(2*besselj(1,paramD)./(besselj(0,paramD).*paramD));
                    elseif rh/deltaN>15            %boundary layer
                        fnu=(1-1i)*deltaN/2/rh;
                        fk=(1-1i)*deltaK/2/rh;
                        fD=(1-1i)*deltaD/2/rh;
                    else                            %intermediate-interpolation
                        paramNU=(1i-1).*2*rh./deltaN; %parameter for convenicence
                        paramK=(1i-1).*2*rh./deltaK;
                        paramD=(1i-1).*2*rh./deltaD;
                        
                        fnu1=2*besselj(1,paramNU)./(besselj(0,paramNU).*paramNU);
                        fk1=2*besselj(1,paramK)./(besselj(0,paramK).*paramK);
                        fD1=2*besselj(1,paramD)./(besselj(0,paramD).*paramD);
                        
                        fnu2=(1-1i)*deltaN/2/rh;
                        fk2=(1-1i)*deltaK/2/rh;
                        fD2=(1-1i)*deltaD/2/rh;
                        
                        fnu=fnu1+(rh-12.5)/(15-12.5)*(fnu2-fnu1);
                        fk=fk1+(rh-12.5)/(15-12.5)*(fk2-fk1);
                        fD=fD1+(rh-12.5)/(15-12.5)*(fD2-fD1);
                    end
                    FD(i)=1-fD;
                    Falpha(i)=1-fk;
                    Fv(i)=1-fnu;
                end
            elseif isa(obj,'TA_Stack')||isa(obj,'TA_HX')
                for i=1:length(obj.X)
                    [ nu, alpha, ~, ~, Pr, Sc] = Mixture_Properties(P_m,obj.Temperature(i),dry_switch,M_Array);
                    deltaN=(2*nu/Omega)^0.5;
                    deltaK=(2*alpha/Omega)^0.5;
                    deltaD=(Pr/Sc)^0.5*deltaK;
                    
                    FD(i)=1-obj.f_function(deltaD);
                    Falpha(i)=1-obj.f_function(deltaK);
                    Fv(i)=1-obj.f_function(deltaN);
                end
            end
        end
        
        %% A function which changes the value of a protected property
        % this function is mainly important to change and bypass protected properties which are children of TA_Component and it has access to them
        function Change_Par_Value_Through_TA_Component(obj,info_array,parameter_case,value)
            switch parameter_case
                case 3  % e.g. {Duct1, Length}
                    obj.(info_array{2}) = value;
                case 4  % e.g. {Duct1, Pressure, 1}
                    obj.(info_array{2})(info_array{3}) = value;
                case 5  % e.g. {1, Duct1, Length}
                    obj.(info_array{3}) = value;
                case 6  % e.g. {1, Duct1, Pressure, 1}
                    obj.(info_array{3})(info_array{4}) = value;
            end
        end

    end
    
    methods (Abstract)
        run_component(obj,P_U_T_in,H_in,locstart)
        Calculate_Derived(obj)
        get_gas_area(obj)

    end
end

