classdef TA_Circular_HX<TA_HX
    %Describes a HX with circular pores

    %% dependent geometrical properties
    properties (SetObservable=true,Dependent) % dependent properties
        Area     % HX area       [m^2]
        A_Solid  % Area covered by solid [m^2]
        Gas_Area_Ratio (1,1) double{mustBeGreaterThan(Gas_Area_Ratio,0)...
            ,mustBeLessThan(Gas_Area_Ratio,1)}
    end
    %% independent properties
    properties (Hidden) % independent properties
    % parameters which keep values and connects to the other parameters:
        Area_val = 0.1
        A_Solid_val = 0;
        Gas_Area_Ratio_val (1,1) double{mustBeGreaterThan(Gas_Area_Ratio_val,0)...
            ,mustBeLessThan(Gas_Area_Ratio_val,1)} = 0.9
    end

    properties (SetObservable=true)
        Diameter (1,1) double{mustBeNonnegative} = 0.1 % Diameter of pores   [m]  
    end
   
    methods
        %% constructor
        function obj = TA_Circular_HX(name,system,length,D,A,A_solid,solid,heat,varargin)
            obj@TA_HX(name,system,length,solid,heat,varargin{:})
            obj.Diameter=D;
            obj.Area_val=A;
            obj.A_Solid_val=A_solid;
        end
        
        %% set method for area
        function set.Area(obj,value)
            obj.Area_val=value;
            obj.A_Solid_val=value*(1-obj.Gas_Area_Ratio_val);
        end
        %% get method for area
        function value=get.Area(obj)
             value=obj.Area_val;
        end

        %% set method for A_solid
        function  set.A_Solid(obj,value)
            if value>=obj.Area_val/2
                error('walls too thick')
            end
            obj.A_Solid_val=value;
            obj.Gas_Area_Ratio_val=1-value/obj.Area_val;
        end
        %% get method for A_Solid
        function value=get.A_Solid(obj)
             value=obj.A_Solid_val;
        end

   %% set method for Porosity
        function  set.Gas_Area_Ratio(obj,value)
            if value<=0.5
                error('walls too thick')
            end
            obj.Gas_Area_Ratio_val=value;
            obj.A_Solid_val=(1-value)*obj.Area_val;
        end
        %% get method for Gas_Area_Ratio
        function value=get.Gas_Area_Ratio(obj)
             value=obj.Gas_Area_Ratio_val;
        end

        %% geometrical functions
        %this function  assists in averaging a property based
        %on the thermal penetration depth and geometry
        %source:delta E.C version 6.4b2.7 user's guide, equations
        %10.59-10.60
        function  f=f_function(obj,delta)
            rh=obj.Diameter/4;
            if rh/delta<12.5              %small enough for bessel
                param=(1i-1).*2*rh./delta; %parameter for convenicence
                
                f=2*besselj(1,param)./(besselj(0,param).*param);
            
            elseif rh/delta>15            %boundary layer
                f=(1-1i)*delta/2/rh;
            else                            %intermediate-interpolation
                param=(1i-1).*2*rh./delta; 
                f1=2*besselj(1,param)./(besselj(0,param).*param);
                f2=(1-1i)*delta/2/rh;
                f=f1+(rh-12.5)/(15-12.5)*(f2-f1);
            end
        end
        function epsilon=eps_function(obj,~,deltas,~)
            rh=obj.Diameter/4;
            epsilon=1/tanh((1+1i)*rh/deltas);
        end
        %% solid temperature calculation
        function T_solid=Calculate_Solid_Temperature(obj)
            % this function calculates the solid temperature based on
            % DEC-delta E.C version 6.4b2.7 user's guide, section 10.7.1
            
            %collect properties from object and system
            p_m=obj.System_H.P_m;
            T_m=obj.Temperature(end);
            rh=obj.Diameter/4;
            Array=obj.Mixture_Array;
            dryswitch=obj.Dry_Switch;

            % if strcmp(obj.Inert,"null") % choosing whether to take Array from system or component
            %     Array=obj.System_H.Mixture_Array;
            % else
            %     Array=obj.Mixture_Array;
            % end
            % if strcmp(obj.Dry_Switch,"null") % choosing whether to take Dry Switch from system or component
            %     dryswitch=obj.System_H.Dry_Switch;
            % else
            %     dryswitch=obj.Dry_Switch;
            % end
            
            omega=obj.System_H.Frequency*2*pi;
            
            %calculate properties
            [ ~, alpha, ~, ~, ~, ~, ~, ~, ~,~,k] = Mixture_Properties(p_m,T_m,dryswitch,Array);
            
            zeta= abs(obj.Velocity(end)/(obj.Area_val-obj.A_Solid_val))/omega;              %acoustic displacement
            xeff=min(2*zeta,obj.Length);
            if xeff == 0  % if velocity is 0, xeff is effectivly 0
                xeff = 1E-13;
            end
            
            deltak=(2*alpha/omega)^0.5;
            yeff=min(deltak,rh);                                                %effective distance from the plate
            h=k/yeff;                                                           %approximate convection coefficient
            T_solid=T_m+obj.Heat_Input/(h*(obj.Area_val-obj.A_Solid_val)*xeff/rh);      %DEC eq 10.113
            
        end
    end
    
end