function [success, collectedvals, vals] = ChangeSmoothSmart( ...
    MySystem, changeparams, newvalues, n_initial, ratio, min_step_size, ...
    Initialguessvalue, Targetvalues, varargin)

% ChangeSmoothSmart - Gradually adjusts system parameters to reach target values.
%
% This function iteratively modifies one or more parameters of `MySystem`, using 
% an adaptive step size approach to prevent divergence. The process continues 
% until all specified parameters reach their exact target values.
%
% INPUTS:
%   MySystem         - (Object) The system object that contains methods for 
%                      evaluating and modifying parameters.
%   changeparams     - (Cell array) Names of the parameters to be modified.
%   newvalues        - (Array) Target values for each parameter in `changeparams`.
%   n_initial        - (Scalar) Initial number of steps to reach `newvalues`.
%   ratio           - (Scalar) Step size adjustment factor (>1 increases, <1 decreases).
%   min_step_size    - (Scalar) Minimum allowed step size to prevent infinite loops.
%   Initialguessvalue - (Array) Initial guess values for running the system.
%   Targetvalues     - (Array) Target output values for system convergence.
%   collectedparams  - (Optional, Cell array) Names of additional parameters to 
%                      track and store during the process.
%
% OUTPUTS:
%   success         - (Boolean) True if successful, False if failed.
%   collectedvals   - (Array) History of collected parameters (if applicable).
%   vals           - (Array) History of modified parameter values during iterations.
%

%%  Initial System Check: Ensure Convergence 
try
    [Currentguessvalue, ~, ~, ~, ~] = MySystem.Run_System_GT( ...
        Initialguessvalue, Targetvalues, 'Not_Smart');
catch
    error('System is not converged at the initial point.');
end

%%  Parameter Initialization 
nparams = length(changeparams);
vals = [];  % Store intermediate values of parameters

% Get initial parameter values from the system
Initial_values = arrayfun(@(j) MySystem.Evaluate_Property(changeparams{j}), 1:nparams);

% Compute baseline step sizes for gradual adjustments
initial_step = (newvalues - Initial_values) / n_initial;
CurrentStepSizeRatio = 1;  % Initialize step size ratio
Currentvalues = Initial_values;  % Start from initial values

% Convergence tracking variables
Failed = false;
laststep = false;
i = 1;  % Iteration counter

%%  Preallocate Storage for Collected Parameters (if applicable) 
if exist('collectedparams', 'var') && ~isempty(collectedparams)
    n_collected = length(collectedparams);
    collectedvals = nan(1e3, n_collected);  % Preallocate with NaNs
else
    collectedvals = [];
end

%%  Iterative Adjustment Process 
while ~all(Currentvalues == newvalues) && ~Failed
    fprintf('Iteration %d\n', i);
    oldGuessvalue = Currentguessvalue;  % Store previous guess value

    % Compute next parameter values
    nextvalues = Currentvalues + initial_step .* CurrentStepSizeRatio;

    % Ensure no overshooting at the last step  
    if abs(nextvalues - Initial_values) >= abs(newvalues - Initial_values)
        laststep = true;
        nextvalues = newvalues;  % Enforce final values at the last step
    end

    % Apply new parameter values to the system
    arrayfun(@(j) MySystem.Change_Property(changeparams{j}, nextvalues(j)), 1:nparams);
    fprintf('StepSizeRatio: %.4f\n', CurrentStepSizeRatio);
    
    % Run system with updated parameters and check convergence
    try
        [Currentguessvalue, Fnorm, Nfev, info] = MySystem.Run_System_GT( ...
            Currentguessvalue, Targetvalues, 'Not_Smart');

        % Collect parameter values if required
        if ~isempty(varargin)
           collectedparams = varargin{1};
           n_collected = length(collectedparams);
           collectedvals(i, :) = arrayfun(@(k) MySystem.Evaluate_Property(collectedparams{k}), 1:n_collected);
        else
           collectedvals = [];
        end


    catch
        info = 0;  % Indicate divergence if the system fails to run
    end
    
    %  Step Size Adjustment Logic 
    if info == 1  % Successful convergence
        if Nfev < 20 && Fnorm < 1e-6
            fprintf('Converged well on this step.\n');
            CurrentStepSizeRatio = CurrentStepSizeRatio * ratio;
        else
            fprintf('Just converged on this step.\n');
        end
        Currentvalues = nextvalues;
        vals = [vals; nextvalues];  % Store successful step values
    else  % Divergence detected
        fprintf('Divergence detected. Reverting changes and reducing step sizes.\n');

        % Revert parameter values to previous state
        arrayfun(@(j) MySystem.Change_Property(changeparams{j}, Currentvalues(j)), 1:nparams);
        
        % Reduce step size ratio (preventing it from going below the minimum)
        CurrentStepSizeRatio = max(CurrentStepSizeRatio / ratio, min_step_size);
        Currentguessvalue = oldGuessvalue;  % Restore previous guess value
        
        % If step size is too small, mark as failed
        if CurrentStepSizeRatio <= min_step_size
            Failed = true;
            break;
        end
        continue;
    end
    
    % Stop loop if last step was reached
    if laststep
        break;
    end

    i = i + 1;  % Increment iteration counter
end

%%  Finalization 
if Failed
    fprintf('Failed: unable to reach target values with a sufficient step size.\n');
    success = false;
else
    success = true;
    fprintf('Succeeded after %d iterations.\n', i);
end

% Trim collected values to the actual number of iterations
if ~isempty(collectedvals)
    collectedvals = collectedvals(1:i, :);
end

end
