function result = ChangeSmooth(MySystem,varargin)
% this function changes a system gradually, while maintaining guesses
% and targets.

%ChangeSmooth(MySystem,{parameter},newValue,n) changes the value of
%"parameter" gradually from it's current value to the new value,
%using n steps .

%ChangeSmooth(MySystem,{parameter,k},newValue,n) changes the value of the
%kth term in "parameter" gradually from it's current value to the new
%value, using n steps


%ChangeSmooth(MySystem,{Component,parameter},newValue,n) changes the value
%of "parameter" in the component described by "component" gradually
%from it's current value to the new value, using n steps.

%ChangeSmooth(MySystem,{Component,parameter,k},newValue,n) changes the
%value of the kth term in "parameter" in the component described by
%"component" gradually from it's current value to the new value, using n
%steps.

%return instructions can be added at the end of the input parameters.
%the different instructions are:

%
%'CollectParams'- collects and returns the values of each parameter
%specified after string. each parameter must be input as an array after the
%string.e.g.
%ChangeSmooth({Component,parameter,k},newValue,n,'CollectParams'...
%,{'Frequency'},{'Duct1','Length'})

%'PlotParams'- plot all the collected parameters

%'CollectSystems'- collect the system at each step as a seperate object
%'CollectData'- collect the DNSQ data from system
%'NoQuestions'- the function will not prompt the user if divergence occurs,
%               will simply cancel
%'NoQuestionsSmart'- the function will not prompt the user if
%               divergence occurs. will attempt to run smartly and then
%               cancel
%'PremitGuessChange' - function will not cause an error if a guessed value
%is changed
%'DispIterations' -display iterations
%'No_Target_Collection' - do not collect targets from system. instead- take
%them from an array that follows the argument


%check that system is converged
if  isempty(MySystem.Pressure)
    error(['Run system while collecting properties before using "ChangeSmooth"', ...
        'make sure system is converged'])
end

param_change=varargin{1};                 %parameter to be change, info
newValue=varargin{2};                     %final value
n=varargin{3};                            %number of iterations
initVal=MySystem.Evaluate_Property(param_change);    %initial value of parameter
%check if parameter is a guess and which guess
nGuess=MySystem.Comparearray(param_change,MySystem.Guesses);
if length(nGuess)>1
    error('cannot change a guessed value with multiple parts')
end
if  nGuess %if parameter is a guess, check if the guess itself is being changed
    tolerance=10^-3;
    switch MySystem.Guesses_RIAP{nGuess}
        case 'R'
            comparison=real(initVal)-real(newValue);
        case 'I'
            comparison=imag(initVal)-imag(newValue);
        case 'P'
            comparison=angle(initVal)-angle(newValue);
        case 'A'
            comparison=abs(initVal)-abs(newValue);
    end
    if abs(comparison)>=tolerance && ~any(strcmp(varargin,'PremitGuessChange'))
        error('cannot change a guessed value.')
    end
end


%name some of the input arguments
%if parameter is a Target, find location in targets. else T=0
T=MySystem.Comparearray(param_change,MySystem.Targets);



%check for instructions
P=false;        %plot
CP=false;       %collect parameters
CS=false;       %collect systems
CD=false;       %collect data
DI=false;       %iteration numbers
NTC=false;      %No target collection
if any(strcmp(varargin,'CollectParams'))
    CP=true;
end
if any(strcmp(varargin,'PlotParams'))
    P=true;
    CP=true;
end
if any(strcmp(varargin,'CollectSystems'))
    CS=true;
end
if any(strcmp(varargin,'CollectData'))
    CD=true;
end
if any(strcmp(varargin,'DispIterations'))
    DI=true;
end
if any(strcmp(varargin,'No_Target_Collection'))
    locNTC=find(strcmp(varargin,'No_Target_Collection'));
    NTC=true;
    Target_Value=varargin{locNTC+1};
end

%current values of guesses and targets
Guess_Value=MySystem.CollectGuessValues;           %initial guess values
if ~NTC
    Target_Value=MySystem.CollectTargetValues;         %target_value
    Target_Value(abs(Target_Value)<1e-3)=0;
end

%collect info about changed parameter
paramname='Changed_Parameter';             %name of parameter
for j=1:length(param_change)
    paramname=[paramname,'_',strrep(num2str(param_change{j}),' ','_')];
end


vals=linspace(initVal,newValue,n); %vector of values
result.(paramname)=vals;

%collect info about output parameters
if CP
    outputparams={};
    outputnames={};
    for i=4:length(varargin)
        if iscell(varargin{i})
            outputparams=[outputparams,varargin(i)];
            name ='Parameter';
            for j=1:length(varargin{i})
                name=[name,'_',replace(num2str(varargin{i}{j}),' ','_')];
            end
            outputnames=[outputnames,{num2str(name)}];
            result.(name)=nan(1,n);
        end
    end
end

%if info should be collected
if CD
    result.Residual_Norm=nan(1,n);
    result.Number_Of_Iterations=nan(1,n);
end

%if System should be collected
if CS
    result.Systems=cell(1,n);
end

%main loop- change system gradualy
oldguess=Guess_Value;
for i=1:n
    if DI
        disp(i)
    end
    %     notify(MySystem,'iteration')
    val=vals(i);

    if T %if changed parameter is a target
        for k=1:length(T)
            CAS=MySystem.Targets_RIAP{T(k)};
            switch CAS
                case 'R'
                    Target_Value(T(k))=real(val);
                case 'I'
                    Target_Value(T(k))=imag(val);
                case 'P'
                    Target_Value(T(k))=angle(val);
                case 'A'
                    Target_Value(T(k))=abs(val);
            end
            if abs(Target_Value(T(k)))<1e-3
                Target_Value(T(k))=0;
            end
        end
    else
        MySystem.Change_Property(param_change,val);
    end
    try
        [X,Fnorm, Nfev,  Info]=MySystem.Run_System_GT(Guess_Value,Target_Value,'Not_Smart');
    catch
        Info=2;
    end
    if Info~=1 %if  diverged
        %% divergence
        if any(strcmp(varargin,'NoQuestions'))
            action=2; %in the "no questions" case- cancel change
        elseif any(strcmp(varargin,'NoQuestionsSmart'))
            action=3;
        else
            % ask user how to proceed
            answer = questdlg(['Algorithm did not converge at step ',num2str(i)...
                ,' of ',num2str(n),'. how would ', 'you like to proceed?'], ...
                'Divergence','Retrace one step', 'Cancel change','Attempt fix','Cancel change');
            % Handle response
            switch answer
                case 'Retrace one step'
                    if i==1
                        action=2;
                    else
                        action=1;
                    end
                case 'Cancel change'
                    action=2;
                case 'Attempt fix'
                    action=3;
            end
        end
        switch action
            case 1 %retrace one step
                val=vals(i-1);
                if T %if changed parameter is a target
                    for k=1:length(T)
                        CAS=MySystem.Targets_RIAP{T(k)};
                        switch CAS
                            case 'R'
                                Target_Value(T(k))=real(val);
                            case 'I'
                                Target_Value(T(k))=imag(val);
                            case 'P'
                                Target_Value(T(k))=angle(val);
                            case 'A'
                                Target_Value(T(k))=abs(val);
                        end
                        if abs(Target_Value(T(k)))<1e-3
                            Target_Value(T(k))=0;
                        end
                    end

                else
                    MySystem.Change_Property(param_change,val);
                end
                try %run system one step behind
                    [~,~, ~,Info]=MySystem.Run_System_GT(Guess_Value,Target_Value,'Smart');
                catch
                    Info=2;
                end
                if Info~=1 % if retrace failed
                    disp('retrace unsuccesfull, cancelling change')
                    val=vals(1);
                    if T %if changed parameter is a target
                        for k=1:length(T)
                            CAS=MySystem.Targets_RIAP{T(k)};
                            switch CAS
                                case 'R'
                                    Target_Value(T(k))=real(val);
                                case 'I'
                                    Target_Value(T(k))=imag(val);
                                case 'P'
                                    Target_Value(T(k))=angle(val);
                                case 'A'
                                    Target_Value(T(k))=abs(val);
                            end
                            if abs(Target_Value(T(k)))<1e-3
                                Target_Value(T(k))=0;
                            end
                        end
                    else
                        MySystem.Change_Property(param_change,val);
                    end
                    Guess_Value=oldguess;
                    MySystem.Run_System_GT(Guess_Value,Target_Value,'Not_Smart','No_Output');
                end
                break
            case 2 %retrace to beginning
                val=vals(1);
                if T %if changed parameter is a target
                    for k=1:length(T)
                        CAS=MySystem.Targets_RIAP{T(k)};
                        switch CAS
                            case 'R'
                                Target_Value(T(k))=real(val);
                            case 'I'
                                Target_Value(T(k))=imag(val);
                            case 'P'
                                Target_Value(T(k))=angle(val);
                            case 'A'
                                Target_Value(T(k))=abs(val);
                        end
                        if abs(Target_Value(T(k)))<1e-3
                            Target_Value(T(k))=0;
                        end
                    end
                else
                    MySystem.Change_Property(param_change,val);
                end
                Guess_Value=oldguess;
                MySystem.Run_System_GT(Guess_Value,Target_Value,'Not_Smart','No_Output');
                break

            case 3 %try smart
                try
                    [X,Fnorm, Nfev,  Info]=MySystem.Run_System_GT(Guess_Value,Target_Value,'Smart');
                catch
                    Info=2;
                end
                if Info~=1
                    disp('Attempt unsuccesfull, canceling change')
                    val=vals(1);
                    if T %if changed parameter is a target
                        for k=1:length(T)
                            CAS=MySystem.Targets_RIAP{T(k)};
                            switch CAS
                                case 'R'
                                    Target_Value(T(k))=real(val);
                                case 'I'
                                    Target_Value(T(k))=imag(val);
                                case 'P'
                                    Target_Value(T(k))=angle(val);
                                case 'A'
                                    Target_Value(T(k))=abs(val);
                            end
                            if abs(Target_Value(T(k)))<1e-3
                                Target_Value(T(k))=0;
                            end
                        end
                    else
                        MySystem.Change_Property(param_change,val);
                        MySystem.Run_System_GT(Guess_Value,Target_Value,'Not_Smart','No_Output');

                    end
                    Guess_Value=oldguess;
                    MySystem.Run_System_GT(Guess_Value,Target_Value,'Not_Smart','No_Output');
                    break
                end

        end
    end
    %% convergence
    Guess_Value=X;
    if CD %if data should be collected
        result.Residual_Norm(i)=Fnorm;
        result.Number_Of_Iterations(i)=Nfev;
    end
    if CP %if parameter should be collected
        for j=1:length(outputparams)
            result.(outputnames{j})(i)=MySystem.Evaluate_Property(outputparams{j});
        end
    end

    if CS
        I=num2str(i);
        compnames=MySystem.Components; %new component names
        for k=1:length(MySystem.Components)
            compnames{k}=[compnames{k},'_',I];
        end
        result.Systems{i}=CopySystem(MySystem,['System_',I],compnames);
    end
end
result.Converged=(Info==1);
if P
    for k=1:length(outputparams)
        figure(k+6)
        if max(imag(result.(outputnames{k}))~=0)
            plot(vals,real(result.(outputnames{k})),vals,imag(result.(outputnames{k})))
            legend('real','Imaginary')
        else
            plot(vals,result.(outputnames{k}))
        end
        title(strrep(outputnames{k},'_',' '))
        xlabel(strrep(paramname,'_',' '))
        ylabel('Parameter')
    end
end

end
