classdef  TA_Union<TA_Component
    % any type of target in a TA system

    properties (SetObservable=true)
        %% unprotected properties
        Length=0               % length [m] just for plotting
        N_Sol_Points=2         % number of points in solution
        Branchnum              % reference to branch number
        % whether or not the split is jacketed. if it is jacketed the total
        % power ratio is ignored
        Jacket(1,1) logical =0
        Pressure_Difference=nan;
        Temperature_Difference=nan;
    end

    methods
        function obj = TA_Union(name,system,branchnumber)
            obj@TA_Component(name,system)
            obj.Branchnum=branchnumber;
        end
        %% main method
        function [PUT_After,H_after]=run_component(obj,P_U_T_in,H_before,locstart)
            % this function runs the component, getting the variables at the
            %begining as an input, and the variables at the end as an output
            if length(obj.System_H.Branches)<obj.Branchnum
                error(['reference to a nonexisting branch number ' num2str(obj.Branchnum)])
            end
            InBranch=obj.System_H.Branches_H{obj.Branchnum};
            if isnan(InBranch.H_end)
                error('Seems like Union is Placed before Split')
            end
            H_branch=InBranch.H_end;

            if isempty((InBranch.Components_H))
                P_branch=InBranch.Begin(1);
                U_branch=InBranch.Begin(2);
                T_branch=InBranch.Begin(3);
            else
                Union_Component=InBranch.Components_H{end};
                P_branch=Union_Component.Pressure(end);
                T_branch=Union_Component.Temperature(end);
                U_branch=Union_Component.Velocity(end);
            end

            P_After=P_U_T_in(1);
            U_after=P_U_T_in(2)+U_branch;
            T_After=P_U_T_in(3);

            if obj.Jacket
                H_after=0.5*real(P_After.*conj(U_after));
            else
                H_after=H_before+H_branch;
            end

            obj.Pressure_Difference=(P_After-P_branch)/abs(P_After);
            obj.Temperature_Difference=T_After-T_branch;

            obj.X=[locstart;locstart+obj.Length];
            obj.Pressure=[P_U_T_in(1),P_After];
            obj.Velocity=[P_U_T_in(2),U_after];
            obj.Temperature=[P_U_T_in(3),T_After];
            PUT_After=[P_After,U_after,T_After];
        end

        %% set method to keep nsolpoints constant
        function set.N_Sol_Points(~,val)
            if val ~= 2
                error('N_Sol_Points for a Union must be 2')
            end
        end
        %% calculate derived properties
        %this function calculates the derived properties (mass flux,
        %acoustic power, total power) based on the pressure, temperature
        % and additional parameters
        function Calculate_Derived(obj)
            if isempty(obj.Pressure)
                error('system must be run before data is collected')
            end
            obj.Acoustic_Power=0.5*real(obj.Pressure.*conj(obj.Velocity));

            %calculate total power
            if obj.Jacket
                obj.Total_Power=obj.Acoustic_Power;
                if obj.Location>1&&obj.Total_Power(1) ~=obj.System_H.Components_H{obj.Location-1}.Total_Power(end)
                    warning('jacket induces jump in acoustic power. consider removing or adding a jacktarget')
                end
            elseif obj.Location>1
                InBranch=obj.System_H.Branches_H{obj.Branchnum};
                H_branch=InBranch.H_end;
                TP1=obj.System_H.Components_H{obj.Location-1}.Total_Power(end);
                obj.Total_Power=[TP1;TP1+H_branch];
            else
                InBranch=obj.System_H.Branches_H{obj.Branchnum};
                H_branch=InBranch.H_end;
                TP1=obj.Acoustic_Power(1);
                obj.Total_Power=[TP1;TP1+H_branch];
            end
            obj.Mass_Flux=0*obj.X;

            %run branch while collecting data
        end
        %% addition to Empty variables for Targets
        function Empty_Varibles(obj)
            obj.Empty_Varibles@TA_Component;
            obj.Pressure_Difference=nan;
            obj.Temperature_Difference=nan;
        end
        %% get gas area
        function gas_area=get_gas_area(~)
            gas_area=NaN;
        end

    end
end

