classdef TA_System <matlab.mixin.Copyable
    % Wet thermoacoustic system

    properties (SetObservable=true)
        %system mean pressure [Pa]
        P_m (1,1)double{mustBeLessThan(P_m,1.2e7)...
            ,mustBeGreaterThan(P_m,9e4)}=1e5

        %system frequncy [Hz]
        Frequency (1,1)double{mustBeLessThan(Frequency,1e6),...
            mustBeGreaterThan(Frequency,0.01)}=100

        Dry_Switch (1,1) double    %1 for dry, 0 for wet

        %oscillating pressure ,oscillating velocity and temperature before
        %the first component. dimensions are [pa] [m/s] [k].
        Begin (1,3) double
    end

    properties (AbortSet=true,SetObservable=true, Dependent)
        % parameters for calculation of mixture properties
        Inert      ... %string with inert component
            {mustBeA(Inert,["cell","string","char"])}
        Reactive   ... %string with reactive component
            {mustBeA(Reactive,["cell","string","char"])}
        Mixture_Array  cell %a cell array with parameters for calculation of mixture properties
    end

    properties (Hidden,AbortSet=true) % independent properties
        % parameters which keep values and connects to the other parameters:
        Inert_val     ... %string with inert component
            {mustBeA(Inert_val,["cell","string","char"])} =""
        Reactive_val  ... %string with reactive component
            {mustBeA(Reactive_val,["cell","string","char"])}="Water"
        Mixture_Array_val  cell

    end

    properties (SetAccess={?TA_Component,?TA_Branch,?TA_Link},SetObservable=true)
        %these are the parameters protected from user access

        Guesses (1,:)cell              % a vector of guesses
        Targets (1,:)cell              % a vector of targets
        X               (:,1) double   %location [m]
        Pressure        (:,1) double   %pressure amlitude [pa]
        Velocity        (:,1) double   %velocity amplitude [m/s]
        Temperature     ...            %temperature [k]
            (:,1) double{mustBeNonnegative}

        Acoustic_Power  (:,1) double   %the acoustic power []
        Total_Power     (:,1) double   %the total power []
        Mass_Flux       (:,1) double   %the mass flux []
        Name            (1,:) char     %string containing name
        Components      (1,:) cell     % a list of component names
        Branches        (1,:) cell     % a list of Branch names
        Links           (1,:) cell     % a list of Link names
        %assisting parameters for  Guesses and targets, indicating real
        %part,imaginary part, absolute value or phase
        Guesses_RIAP    (1,:) cell
        Targets_RIAP    (1,:) cell
    end

    properties
        Components_H (1,:) cell %handles to actual components
        Branches_H   (1,:) cell %handles to actual Branches
        Links_H      (1,:) cell %handles to actual Links
        %assisting parameters for  Guesses and targets, indicating format
        % the cases are:
        %1- a system property with no index ('P_m')
        %2- a system property with and index('Begin',1)
        %[3,I]- a component property with no index ('Duct1','Length')
        %       where I indicates component location in system
        %[4,I]- a component property with  index ('Duct1','Pressure',1)
        %       where I indicates component location in system
        %[5,I]- a component property with branch (1,'Duct1','Length')
        %         for component I in branch J.
        %[6,I]- a component property with branch and index
        %         (1,'Duct1','Pressure',1) for component I in branch J.
        Guesses_ID (1,:) cell
        Targets_ID (1,:) cell

        %variables for Mixture&Solid Properties_ODE
        saved_p = NaN
        saved_T = NaN
        saved_m_properties=NaN(16,1);

        %Assisting property for checking chain links errors
        Chain_Link_Error_Flag (1,1) double = 0

    end

    properties (SetAccess=private)
    % Animation Data
    Animation_Pressure (:,:) double   % Pressure data [Pa]
    Animation_Velocity (:,:) double   % Velocity data [m/s]
    Animation_Temperature (:,:) double % Temperature data [K]
    Animation_Concentration (:,:) double % Concentration data (if applicable)
    Animation_x_axis (1,:) double     % X-axis positions [m]
    end


    methods
        %% constructor
        function obj = TA_System(name,p_m,frequency,inert,reactive,dry_switch,begin)
            obj.Name=name;
            obj.P_m = p_m;
            if ischar(inert)
                inert=string(inert);
            end
            if ischar(reactive)
                reactive=string(reactive);
            end
            obj.Inert=inert;
            obj.Reactive=reactive;
            obj.Frequency=frequency;
            obj.Dry_Switch=dry_switch;
            obj.Begin=begin;
            obj.Components={};
            obj.Components_H={};
            obj.Branches={};
            obj.Branches_H={};
            obj.Links={};
            obj.Links_H={};
            obj.Guesses={};
            obj.Targets={};
            obj.Guesses_ID={};
            obj.Targets_ID={};
            obj.Guesses_RIAP={};
            obj.Targets_RIAP={};
        end
        %% set & get methods for inert and reactive components, as well as dryswitch
        % Set method for Inert:
        function set.Inert(obj,value)
            if ischar(value)
                value=string(value);
            elseif iscell(value)
                value{1}=string(value{1});
                value{3}=string(value{3});
            end
          %  old_val = obj.Inert_val;
            obj.Inert_val=value;
            if ~isempty(obj.Reactive)
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end

            %    % Change the synchornized components properties:
            %    not_changed_msg = '';
            %    unintent_synchronized_msg = '';
            %    for i=1:length(obj.Components)
            %        if (old_val == obj.Components_H{i}.Inert) && (obj.Reactive == obj.Components_H{i}.Reactive)
            %            obj.Components_H{i}.Inert = value;
            %        else
            %            % Components which are not changed:
            %            if isempty(not_changed_msg)
            %                not_changed_msg = obj.Components{i};
            %            else
            %                not_changed_msg = [not_changed_msg, ', ', obj.Components{i}];
            %            end
            %            % Components that will also synchronize unintentionally with the system after it's change:
            %            if (value == obj.Components_H{i}.Inert) && (obj.Reactive == obj.Components_H{i}.Reactive)
            %                if isempty(unintent_synchronized_msg)
            %                    unintent_synchronized_msg = obj.Components{i};
            %                else
            %                    unintent_synchronized_msg = [unintent_synchronized_msg, ', ', obj.Components{i}];
            %                end
            %            end
            %        end
            %    end
            %    if ~isempty(not_changed_msg)
            %        % Code format:
            %        warning(['The following components Inert properties were not changed:',...
            %                 char(10),not_changed_msg])
            %        
            %        % GUI format:
            %        %uialert(Figure,['The following components Inert properties were not changed:',...
            %        %char(10),not_changed_msg]...
            %        %,'Message','Icon','info','Modal',false)
            %    end
            %
            %    if ~isempty(unintent_synchronized_msg)
            %        % Code format:
            %        warning(['Pay attention that the following components synchronized unintentionally with the system:',...
            %                 char(10),unintent_synchronized_msg])
            %    end
        end
        % Get method for Inert:
        function value=get.Inert(obj)
            value=obj.Inert_val;
        end

        % Set method for Reactive:
        function set.Reactive(obj,value)
            if ischar(value)
                value=string(value);
            elseif iscell(value)
                value{1}=string(value{1});
                value{3}=string(value{3});
            end
          %  old_val = obj.Reactive_val;
            obj.Reactive_val=value;
            if ~isempty(obj.Reactive)
                obj.Mixture_Array=collect_properties(obj.Inert,obj.Reactive);
            end
       %     % Change the synchornized components properties:
       %     not_changed_msg = '';
       %     unintent_synchronized_msg = '';
       %     for i=1:length(obj.Components)
       %         if (old_val == obj.Components_H{i}.Reactive) && (obj.Inert == obj.Components_H{i}.Inert)
       %             obj.Components_H{i}.Reactive = value;
       %         else
       %             % Components which are not changed:
       %             if isempty(not_changed_msg)
       %                 not_changed_msg = obj.Components{i};
       %             else
       %                 not_changed_msg = [not_changed_msg, ', ', obj.Components{i}];
       %             end
       %             % Components that will also synchronize unintentionally with the system after it's change:
       %             if (value == obj.Components_H{i}.Reactive) && (obj.Inert == obj.Components_H{i}.Inert)
       %                 if isempty(unintent_synchronized_msg)
       %                     unintent_synchronized_msg = obj.Components{i};
       %                 else
       %                     unintent_synchronized_msg = [unintent_synchronized_msg, ', ', obj.Components{i}];
       %                 end
       %             end
       %         end
       %     end
       %     if ~isempty(not_changed_msg)
       %         % Code format:
       %         warning(['The following components Reactive properties were not changed:',...
       %                  char(10),not_changed_msg])
       %         
       %         % GUI format:
       %         %uialert(Figure,['The following components Reactive properties were not changed:',...
       %         %char(10),not_changed_msg]...
       %         %,'Message','Icon','info','Modal',false)
       %     end
       % 
       %     if ~isempty(unintent_synchronized_msg)
       %         % Code format:
       %         warning(['Pay attention that the following components synchronized unintentionally with the system:',...
       %                  char(10),unintent_synchronized_msg])
       %     end
        end
        % Get method for Reactive:
        function value=get.Reactive(obj)
            value=obj.Reactive_val;
        end

        % Set method for Mixture_Array
        function  set.Mixture_Array(obj,value)
            obj.Mixture_Array_val=value;      
        end
        % Get method for Mixture_Array:
        function value=get.Mixture_Array(obj)
            value=obj.Mixture_Array_val;
        end

        % initializing saved properties when Dry_Switch is modified
        function set.Dry_Switch(obj,value)
            obj.Dry_Switch=value;
            obj.saved_p=NaN;
            obj.saved_T=NaN;
            obj.saved_m_properties=NaN(16,1);
        end

        % A function for reseting all components mixture properties to
        % "null", and then they will become dependent on the system properties
        function Reset_Components_Mix_Prop(obj)
            
            for i=1:length(obj.Components)
                obj.Components_H{i}.Inert = "null";
                obj.Components_H{i}.Reactive = "null";
                obj.Components_H{i}.Dry_Switch = "null";
                obj.Components_H{i}.Mixture_Array = {};
            end
        end

        %% rename function
        function rename(obj,newName)
            for i=1:length(obj.Components)
                obj.Components_H{i}.System=newName;
            end
            for i=1:length(obj.Branches)
                obj.Branches_H{i}.Mother_System=newName;
            end
            for i=1:length(obj.Links)
                obj.Links{i}.System=newName;
            end
            obj.Name=newName;
        end
        %% add component to name list
        %this function is used internaly, shouldn't be used by the user to
        %add a component. a component is added automatically when it is
        %created if the system is selected under "system"
        function addComponent(obj,component)
            if any(strcmp(obj.Components,component))
                error(['A component already exists with the name '...
                    component])
            end
            obj.Components=[obj.Components,component];
        end
        %% add branch to name list
        %this function is used internaly, shouldn't be used by the user to
        %add a branch. a branch is added automatically when it is
        %created if the system is selected under "system"
        function addBranch(obj,Branch)
            if any(strcmp(obj.Branches,Branch))
                error(['A Branch already exists with the name '...
                    Branch])
            end
            obj.Branches=[obj.Branches,Branch];
        end
        %% add branch to name list
        %this function is used internaly, shouldn't be used by the user to
        %add a link. a link is added automatically when it is
        %created if the system is selected under "system"
        function addLink(obj,Link)
            if any(strcmp(obj.Links,Link))
                error(['A Link already exists with the name '...
                    Link])
            end
            obj.Links=[obj.Links,Link];
        end
        %% add a Guess
        function addGuess(obj,varargin)
            % this function adds a guess to the system's guess list

            % addGuess(property)- input is a string with a property name
            % e.g addGuess('P_m') sets P_m as a guess

            % addGuess(property,i) input is a string with property name
            % and an integer index
            % e.g addGuess('Begin',3) sets Begin(3) (temperature at the
            % begining) as a guess.

            % addGuess(component,component_property) input is a string with
            % component name and another one with the component property
            % name.
            % e.g addGuess ('Duct1','length')
            % sets the length of component duct1 as a guess

            % addGuess(component,component_property,i) input is a string with
            % component name, another one with the component property
            % name, and an index

            % addGuess(branch,component,component_property) input is a branch
            % number, a string with component name, and another one with the
            % component property name

            % addGuess(branch,component,component_property,i) input is a branch
            % number, a string with component name, another one with the
            % component property name, and an index

            %if the added  guess is a complex number, the default option
            %would be to add the real part. if you want to add the
            %imaginary part, phase, or absolute value, you need to add a
            %string with the words 'Imag','Phase',or 'Abs' to it.
            % e.g addGuess('Imag',Begin',1) sets imag(Begin(1)) (imaginary part
            % of pressure the begining) as a guess.


            N=length(obj.Guesses);
            %check if guess is real, imaginary, phase or absolute
            switch varargin{1}
                case 'Real'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='R';
                case 'Imag'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='I';
                case 'Phase'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='P';
                case 'Abs'
                    varargin=varargin(2:end);
                    obj.Guesses_RIAP{N+1}='A';
                otherwise
                    obj.Guesses_RIAP{N+1}='R';
            end

            % case 1
            if length(varargin)==1&&ischar(varargin{1})
                try
                    obj.(varargin{1});
                    obj.Guesses_ID{N+1}=1;
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                %case 2
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&isnumeric(varargin{2})
                try
                    obj.(varargin{1});
                    obj.Guesses_ID{N+1}=2;
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                %case 3
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&ischar(varargin{2})
                I=obj.findLoc(varargin{1});
                try
                    obj.Components_H{I}.(varargin{2});
                    obj.Guesses_ID{N+1}=[3,I];
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 4
            elseif length(varargin)==3&&ischar(varargin{1})&&...
                    ischar(varargin{2})&&isnumeric(varargin{3})
                I=obj.findLoc(varargin{1});
                try
                    obj.Components_H{I}.(varargin{2});
                    obj.Guesses_ID{N+1}=[4,I];
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 5
            elseif length(varargin)==3&&isnumeric(varargin{1})&&...
                    ischar(varargin{2})&&ischar(varargin{3})
                Branchnum=varargin{1};
                if Branchnum<1||Branchnum>length(obj.Branches)
                    error(['Attempt to access branch ',num2str(Branchnum)...
                        ' which does not exist'])
                end
                I=obj.Branches_H{Branchnum}.findLoc(varargin{2});
                try
                    obj.Branches_H{Branchnum}.Components_H{I}.(varargin{3});
                    obj.Guesses_ID{N+1}=[5,I];
                catch
                    error(['no property exists with the name ',varargin{3}])
                end
                %case 6
            elseif length(varargin)==4&&isnumeric(varargin{1})&&...
                    ischar(varargin{2})&&ischar(varargin{3})&&isnumeric(varargin{4})
                Branchnum=varargin{1};
                if Branchnum<1||Branchnum>length(obj.Branches)
                    error(['Attempt to access branch ',num2str(Branchnum)...
                        ' which does not exist'])
                end
                I=obj.Branches_H{Branchnum}.findLoc(varargin{2});
                try
                    obj.Branches_H{Branchnum}.Components_H{I}.(varargin{3});
                    obj.Guesses_ID{N+1}=[6,I];
                catch
                    error(['no property exists with the name ',varargin{3}])
                end
            else
                error('wrong format for adding a Guess or target')
            end

            % link validation if exists
            if ~isempty(obj.Links)
                try
                    obj.Added_Guess_Link_Validation(varargin)
                catch me
                    obj.Guesses_ID(N+1) = [];
                    error(['Can not add guess to chosen paramter, for it is linked and elevates the follwing issue:',char(10),me.message])
                end
            end

            % adding the guess
            obj.Guesses{N+1}=varargin;
        end
        %% add a Target
        function addTarget(obj,varargin)
            % this function adds a Target to the system's Target list

            % addTarget(property)- input is a string with a property name
            % e.g addTarget('P_m') sets P_m as a Target

            % addTarget(property,i) input is a string with property name
            % and an integer index
            % e.g addTarget('Begin',3) sets Begin(3) (temperature at the
            % begining) as a Target.

            % addTarget(component,component_property) input is a string with
            % component name and another one with the component property
            % name.
            % e.g addTarget ('Duct1','length')
            % sets the length of component duct1 as a Target

            % addTarget(component,component_property,i) input is a string with
            % component name, another one with the component property
            % name, and an index

            % addTarget(branch,component,component_property) input is a branch
            % number, a string with component name, and another one with the
            % component property name

            % addTarget(branch,component,component_property,i) input is a branch
            % number, a string with component name, another one with the
            % component property name, and an index

            %if the added  Target is a complex number, the default option
            %would be to add the real part. if you want to add the
            %imaginary part, phase, or absolute value, you need to add a
            %string with the words 'Imag','Phase',or 'Abs' to it.
            % e.g addTarget('Imag',Begin',1) sets imag(Begin(1)) (imaginary part
            % of pressure the begining) as a Target.


            N=length(obj.Targets);
            %check if Target is real, imaginary, phase or absolute
            switch varargin{1}
                case 'Real'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='R';
                case 'Imag'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='I';
                case 'Phase'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='P';
                case 'Abs'
                    varargin=varargin(2:end);
                    obj.Targets_RIAP{N+1}='A';
                otherwise
                    obj.Targets_RIAP{N+1}='R';
            end

            % case 1
            if length(varargin)==1&&ischar(varargin{1})
                obj.Targets_ID{N+1}=1;
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                %case 2
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&isnumeric(varargin{2})
                try
                    obj.(varargin{1});
                catch
                    error(['no property exists with the name ',varargin{1}])
                end
                obj.Targets_ID{N+1}=2;
                %case 3
            elseif length(varargin)==2&&ischar(varargin{1})...
                    &&ischar(varargin{2})
                I=obj.findLoc(varargin{1});
                obj.Targets_ID{N+1}=[3,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 4
            elseif length(varargin)==3&&ischar(varargin{1})&&...
                    ischar(varargin{2})&&isnumeric(varargin{3})
                I=obj.findLoc(varargin{1});
                obj.Targets_ID{N+1}=[4,I];
                try
                    obj.Components_H{I}.(varargin{2});
                catch
                    error(['no property exists with the name ',varargin{2}])
                end
                %case 5
            elseif length(varargin)==3&&isnumeric(varargin{1})&&...
                    ischar(varargin{2})&&ischar(varargin{3})
                Branchnum=varargin{1};
                if Branchnum<1||Branchnum>length(obj.Branches)
                    error(['Attempt to access branch ',num2str(Branchnum)...
                        ' which does not exist'])
                end
                I=obj.Branches_H{Branchnum}.findLoc(varargin{2});
                try
                    obj.Branches_H{Branchnum}.Components_H{I}.(varargin{3});
                catch
                    error(['no property exists with the name ',varargin{3}])
                end
                obj.Targets_ID{N+1}=[5,I];
                %case 6
            elseif length(varargin)==4&&isnumeric(varargin{1})&&...
                    ischar(varargin{2})&&ischar(varargin{3})&&isnumeric(varargin{4})
                Branchnum=varargin{1};
                if Branchnum<1||Branchnum>length(obj.Branches)
                    error(['Attempt to access branch ',num2str(Branchnum)...
                        ' which does not exist'])
                end
                I=obj.Branches_H{Branchnum}.findLoc(varargin{2});
                try
                    obj.Branches_H{Branchnum}.Components_H{I}.(varargin{3});
                catch
                    error(['no property exists with the name ',varargin{3}])
                end
                obj.Targets_ID{N+1}=[6,I];
            else
                error('wrong format for adding a Guess or target')
            end
            obj.Targets{N+1}=varargin;
        end
        %% run system without targets or guesses
        function  Hout=runSystem(obj,collectdata,varargin)
            %this function runs a thermoacoustic system
            %collectdata determines whether data is collected from the system,
            %running with different total power is optional by adding a
            %second parameter
           
            %check varargin
            if isempty(varargin)
                Hdifferent=nan;
            elseif strcmp(varargin{1},'Hdifferent')&&isnumeric(varargin{2})
                Hdifferent=varargin{2};
            else
                error('Wrong format for runsystem')
            end

            %update branches weather to collect data
            for i=1:length(obj.Branches)
                obj.Branches_H{i}.Calc_Derived=collectdata;
            end

            %empty all variables
            obj.Pressure=[];
            obj.Velocity=[];
            obj.Temperature=[];
            obj.X=[];
            obj.Mass_Flux=[];
            obj.Total_Power=[];
            obj.Acoustic_Power=[];
            
            myLoc=0;
            %Pressure, Velocity, Temperature
            PUT=obj.Begin;
            %total power
            if isnan(Hdifferent)
                H=0.5*real(PUT(1)*conj(PUT(2)));
            else
                H=Hdifferent;
            end

            %run and collect data if requested
            if ~collectdata
                for i=1:length(obj.Components)
                    %run component
                    [PUT,H]=obj.Components_H{i}.run_component(PUT,H,myLoc);
                    %update location
                    myLoc=myLoc+obj.Components_H{i}.Length;
                end
            else
                for i=1:length(obj.Components)
                    %run component
                    [PUT,H]=obj.Components_H{i}.run_component(PUT,H,myLoc);
                    %update location
                    myLoc=myLoc+obj.Components_H{i}.Length;
                    %calculate derived parameters
                    obj.Components_H{i}.Calculate_Derived
                    %collect data
                    obj.Pressure=[obj.Pressure;obj.Components_H{i}.Pressure];
                    obj.Velocity=[obj.Velocity;obj.Components_H{i}.Velocity];
                    obj.Temperature=[obj.Temperature;obj.Components_H{i}.Temperature];
                    obj.X=[obj.X;obj.Components_H{i}.X];
                    obj.Mass_Flux=[obj.Mass_Flux;obj.Components_H{i}.Mass_Flux];
                    obj.Total_Power=[obj.Total_Power;obj.Components_H{i}.Total_Power];
                    obj.Acoustic_Power=[obj.Acoustic_Power;obj.Components_H{i}.Acoustic_Power];
                end
            end
            Hout=H;
        end
        %% run system with guesses and targets
        function [X,Fnorm, Nfev,Info, errorInf]=Run_System_GT(obj,guessvalues,targetvalues,varargin)
            %this function runs the system with guesses and targets

            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues)
            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues,'Smart')
            %[X,Fvec, Nfev,  Info]=obj.Run_System_GT(guessvalues,targetvalues,'Not_Smart')

            %input parameters- guessvalues, targetvalues self explanatory
            %                  if 'Smart' is input, system will attempt several
            %                  ways of converging in case of divergence.
            %                  if  'Not_Smart' no other convergance ways will
            %                  be attempted
            %                  %if neither is input, system will prompt user to
            %                  decide.
            %                  % if "No_Output" is printed the system will
            %                  not print any output

            if length(obj.Guesses)~=length(obj.Targets)||length(obj.Guesses)...
                    ~=length(guessvalues)||length(guessvalues)~=length(targetvalues)
                error('system must have the same number of guesses, targets and guess values')
            end

            % check if the derived properties needs to be calculated during run
            Calc_deriverd=0;
            for i=1:length(obj.Targets)
                switch obj.Targets_ID{i}(1)
                    case{3,4}
                        if max(strcmp(obj.Targets{i}{2},...
                                {'Acoustic_Power','Total_Power','Mass_Flux'}))
                            Calc_deriverd=1;
                            break
                        end
                    case{5,6}
                        if max(strcmp(obj.Targets{i}{3},...
                                {'Acoustic_Power','Total_Power','Mass_Flux'}))
                            Calc_deriverd=1;
                            break
                        end
                end
            end
            if Calc_deriverd
                warning(['running system with a derived property',...
                    '(acoustic power, total power or mass flux) as a guess',...
                    'or target can slow the run. consider redesigning the',...
                    'guesses and targets'])
            end

            %run Hybrid_Powel
            func=@(g)(obj.G2T(g,targetvalues,Calc_deriverd));
            [Fvec, Nfev, X, Info,errorInf] = Hybrid_Powel(func,guessvalues);
            if ~isempty(errorInf)
                switch errorInf.I
                    case 'Mixture_properties:aboveB'
                        warning(['system went above boiling temperature at'...
                            ,'least once during this run, guess values were [',...
                            num2str(errorInf.X), ']. consider changing ',...
                            'the initial guess. to investigate the error',...
                            ' message, run the system with these values.'])
                    otherwise
                        warning(['system encountered an error at'...
                            ,'least once during this run, guess values were [',...
                            num2str(errorInf.X), ']. consider changing ',...
                            'the initial guess. to investigate the error',...
                            ' message, run the system with these values.'])
                end
            end

            if Info==0
                fprintf('error, check input parameters')
            elseif Info>1
                if any(strcmp(varargin,'Smart'))
                    smart=true;
                elseif any(strcmp(varargin,'Not_Smart'))
                    smart=false;
                else
                    %check if should contiunue smartly or not
                    answer = questdlg(['Algorithm did not converge. would ', ...
                        'you like algorithm to attempt fixing it?'], ...
                        'Divergence', 'Yes','No','Yes');
                    % Handle response
                    switch answer
                        case 'Yes'
                            smart=true;
                        case 'No'
                            smart=false;
                    end
                end
                if smart %smart attempts
                    % try rerunning a few times
                    Fnorm=norm(Fvec,2);
                    if Fnorm<2
                        for i=1:3
                            [Fvec, Nfev, X, Info] = Hybrid_Powel(func,X);
                            if Info==1
                                break
                            end
                        end
                    end
                    %try relaxing Xtol
                    Fnorm=norm(Fvec,2);
                    if Fnorm<0.1&&~Info~=1
                        [Fvec, Nfev, X, Info] = Hybrid_Powel(func,X,1e-4,100,250,1e-12);
                    end
                    %try some random attempts around the original guess
                    if Info~=1
                        for i=1:20
                            Xtry=guessvalues.*(rand(1,length(X))*0.1+0.95);
                            try
                                [Fvec, Nfev, X, Info] = Hybrid_Powel(func,Xtry);
                            catch
                                Info=0;
                            end
                            if Info==1
                                break
                            end
                        end
                    end

                end
            end
            Fnorm=norm(Fvec,2);
            %post processing
            if Info==1 % if converged
                obj.G2T(X,targetvalues,1);
                if ~any(strcmp(varargin,'No_Output'))
                    %print results
                    fprintf('algorithm converged!\n ')
                    fprintf(strrep(['The converged value for guesses is: (' sprintf(' %d,', X) ')'], ',)', ')\n'))
                    fprintf('number of function evaluations is %d\n',Nfev)
                    fprintf('norm of the residuals is %d\n',Fnorm)
                end
            else
                for i=1:length(obj.Components_H)
                    obj.Components_H{i}.Empty_Varibles;
                end
                if ~any(strcmp(varargin,'No_Output'))
                    fprintf('targets did not converge. try changing guess \n')
                end
            end

        end
        %% run system with fsolve instead
        function X=Run_System_GT2(obj,guessvalues,targetvalues,varargin)
            if length(obj.Guesses)~=length(obj.Targets)||length(obj.Guesses)...
                    ~=length(guessvalues)||length(guessvalues)~=length(targetvalues)
                error('system must have the same number of guesses, targets and guess values')
            end

            %run fsolve
            func=@(g)(obj.G2T(g,targetvalues,0));
            X = fsolve(func,guessvalues);
        end
        %%  assisting function for running system with targets and guesses
        function target= G2T(obj,guessvalue,targetvalue,Calc_deriverd)
            %this function gets as an input a guessed value of all guessed
            %variables, runs the system and returns the target value
            %Calc_deriverd determines weather to calculate derived
            %properties

            %collect guesses
            for i=1:length(obj.Guesses)
                switch obj.Guesses_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        new=obj.changeguess(obj.(obj.Guesses{i}{1}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.(obj.Guesses{i}{1})=new;
                    case 2
                        new=obj.changeguess(obj.(obj.Guesses{i}{1})(obj.Guesses{i}...
                            {2}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.(obj.Guesses{i}{1})(obj.Guesses{i}{2})=new;
                    case 3
                        new=obj.changeguess(obj.Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{2}),guessvalue(i),obj.Guesses_RIAP{i});
                        obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}{2})=new;
                    case 4
                        new=obj.changeguess(obj.Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{2})(obj.Guesses{i}{3}),guessvalue(i)...
                            ,obj.Guesses_RIAP{i});
                        obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {2})(obj.Guesses{i}{3})=new;
                    case 5
                        new=obj.changeguess(obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{3}),guessvalue(i)...
                            ,obj.Guesses_RIAP{i});
                        obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {3})=new;
                    case 6
                        new=obj.changeguess(obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{3})(obj.Guesses{i}{4}),guessvalue(i)...
                            ,obj.Guesses_RIAP{i});
                        obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {3})(obj.Guesses{i}{4})=new;
                end
            end
            %run system
            obj.runSystem(Calc_deriverd);

            %collect targets
            target=zeros(size(guessvalue));
            for i=1:length(obj.Targets)
                switch obj.Targets_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        target(i)=obj.(obj.Targets{i}{1});
                    case 2
                        target(i)=obj.(obj.Targets{i}{1})(obj.Targets{i}{2});
                    case 3
                        target(i)=obj.Components_H...
                            {obj.Targets_ID{i}(2)}.(obj.Targets{i}{2});
                    case 4
                        target(i)=obj.Components_H...
                            {obj.Targets_ID{i}(2)}.(obj.Targets{i}{2})(obj.Targets{i}{3});
                    case 5
                        target(i)=obj.Branches_H{obj.Targets{i}{1}}.Components_H...
                            {obj.Targets_ID{i}(2)}.(obj.Targets{i}{3});
                    case 6
                        target(i)=obj.Branches_H{obj.Targets{i}{1}}.Components_H...
                            {obj.Targets_ID{i}(2)}.(obj.Targets{i}{3})(obj.Targets{i}{4});
                end
                switch obj.Targets_RIAP{i}(1)
                    case 'R'
                        %real value
                        if targetvalue(i)==0
                            target(i)=real(target(i));
                        else
                            target(i)=real(target(i))/targetvalue(i)-1;
                        end
                    case 'I'
                        %imaginary value
                        if targetvalue(i)==0
                            target(i)=imag(target(i));
                        else
                            target(i)=imag(target(i))/targetvalue(i)-1;
                        end
                    case 'A'
                        %absolute value
                        if targetvalue(i)==0
                            target(i)=abs(target(i));
                        else
                            target(i)=abs(target(i))/targetvalue(i)-1;
                        end
                    case 'P'
                        %phase
                        if targetvalue(i)==0
                            target(i)=angle(target(i));
                        else
                            target(i)=angle(target(i))/targetvalue(i)-1;
                        end
                end
            end

        end
        %% plotting function
        function varargout=PlotSystem(obj,varargin)
            %this function plots the oscilating pressure, oscilating velocity
            %and mean temperature in all the system components

            %preperatory work
            set(0,'DefaultFigureWindowStyle','docked')
            if  any(strcmp(varargin,'forGUI'))
                GUI=true;
            else
                GUI=false;
            end
            if  isempty(obj.Pressure)
                error('system must be run before it is plotted')
            end

            % new code for figures

            % figure 1-2 for normal pressure and velocity plot
            % figure 3 for temperature plot
            % figure 4-6 for derived plot
            % figure 7-8 for Real/Img scaled plot
            % figure 9-10 for Abs/Phase plot
            fs=cell(1,10); 
            as=cell(1,10);
            as_right=cell(1,10); % right double axis

           % creating the figures
            for i=1:10
                try
                    close(i) % delete the figure if existed
                catch
                end
                %if ~(i>3&&~(any(strcmp(varargin,'Derived')))) % skip 4-6 if derived not chosen
                fs{i}=figure(i);
                as{i}=axes(fs{i});
                %end
                if i>6
                    % creating the right axis
                    hold on
                    as_right{i}=axes(fs{i},'yaxislocation','right');
                    hold on
                    %set(as_right{i},'color','none')
                    set(as{i},'color','none')
                end
 
                hold on
                fs{i}.Visible='Off';
            end
            
            legends=cell(1,length(obj.Components));
            legends_complex=cell(4,length(obj.Components));
            legends_list=[];

            targetcounter=0; %counts number of targets in plot
            % set some properties for all plots
            set(0, 'DefaultAxesFontWeight', 'bold', ...
                'DefaultAxesFontSize', 10, ...
                'DefaultAxesFontWeight', 'bold', ...
                'DefaultAxesTitleFontWeight', 'bold', ...
                'DefaultAxesTitleFontSizeMultiplier',2) ;
            set(0, 'DefaultLineLineWidth', 4);

            for i=1:length(obj.Components) %run over all components
                if ~isa(obj.Components_H{i},'TA_rget')
                %plot pressure
                    % creating the graphs
                    plot7a=plot(as{7},obj.Components_H{i}.X,real(obj.Components_H{i}.Pressure),'-');
                    plot7b=plot(as_right{7},obj.Components_H{i}.X,imag(obj.Components_H{i}.Pressure),'-..');
                    %make sure both plots have the same style
                    set(plot7b,'color',get(plot7a,'color'))
                    % creating the graphs
                    plot9a=plot(as{9},obj.Components_H{i}.X,abs(obj.Components_H{i}.Pressure),'-');
                    plot9b=plot(as_right{9},obj.Components_H{i}.X,180/pi*angle(obj.Components_H{i}.Pressure),'-..');
                    %make sure both plots have the style
                    set(plot9b,'color',get(plot9a,'color'))
                    % creating the graphs
                    plot1a=plot(as{1},obj.Components_H{i}.X,real(obj.Components_H{i}.Pressure),'-');
                    plot1b=plot(as{1},obj.Components_H{i}.X,imag(obj.Components_H{i}.Pressure),'-..');
                    %make sure both plots have the same color
                    set(plot1b,'color',get(plot1a,'color'))
                %plot velocity
                    % creating the graphs
                    plot8a=plot(as{8},obj.Components_H{i}.X,real(obj.Components_H{i}.Velocity),'-');
                    plot8b=plot(as_right{8},obj.Components_H{i}.X,imag(obj.Components_H{i}.Velocity),'-..');
                    %make sure both plots have the same style
                    set(plot8b,'color',get(plot8a,'color'))
                    % creating the graphs
                    plot10a=plot(as{10},obj.Components_H{i}.X,abs(obj.Components_H{i}.Velocity),'-');
                    plot10b=plot(as_right{10},obj.Components_H{i}.X,180/pi*angle(obj.Components_H{i}.Velocity),'-..');
                    %make sure both plots have the style
                    set(plot10b,'color',get(plot10a,'color'))
                    % creating the graphs
                    plot2a=plot(as{2},obj.Components_H{i}.X,real(obj.Components_H{i}.Velocity),'-');
                    plot2b=plot(as{2},obj.Components_H{i}.X,imag(obj.Components_H{i}.Velocity),'-..');
                    %make sure both plots have the same color
                    set(plot2b,'color',get(plot2a,'color'))

                %plot temperature
                   plot3=plot(as{3},obj.Components_H{i}.X,obj.Components_H{i}.Temperature);
                    %make sure all plots are of the same color
                    set(plot3,'color',get(plot1a,'color'))

                    %create legends array
                    name1=strrep(obj.Components_H{i}.Name,'_',' ');
                    %if any(strcmp(varargin,'Abs'))
                        name2_abs=strcat(name1,' Abs');
                        name3_ang=strcat(name1,' Phase');
                    %else
                        name2_real=strcat(name1,' Real');
                        name3_img=strcat(name1,' Imag');
                    %end
                    legends{i}=name1;
                    legends_complex(:,i)={name2_real,name3_img,name2_abs,name3_ang};
                    %if any(strcmp(varargin,'Scaled')) || any(strcmp(varargin,'Abs'))
                        %if the plot is scaled, plot handles need to be
                        %stored for legend
                        legends_list=[legends_list,[plot1a;plot1b;plot2a;plot2b;...
                                                    plot7a;plot7b;plot9a;plot9b;...
                                                    plot8a;plot8b;plot10a;plot10b]];
                    %end
                    %create a plot of derived functions as well
                    if GUI || any(strcmp(varargin,'Derived'))
                        plot4=plot(as{4},obj.Components_H{i}.X,obj.Components_H{i}.Acoustic_Power);
                        %make sure all plots are of the same color
                        set(plot4,'color',get(plot3,'color'))
                        plot5=plot(as{5},obj.Components_H{i}.X,obj.Components_H{i}.Total_Power);
                        %make sure all plots are of the same color
                        set(plot5,'color',get(plot3,'color'))
                        plot6=plot(as{6},obj.Components_H{i}.X,obj.Components_H{i}.Mass_Flux);
                        %make sure all plots are of the same color
                        set(plot6,'color',get(plot3,'color'))
                    end
                else %if it is a target
                    legends(i)={nan};
                    legends_complex(:,i)={nan;nan;nan;nan};
                    targetcounter=targetcounter+1;
                end
            end
            
            % linking the scaled axes to organize the graphs
            linkaxes([as{7} as_right{7}],'x')
            linkaxes([as{8} as_right{8}],'x')
            linkaxes([as{9} as_right{9}],'x')
            linkaxes([as{10} as_right{10}],'x')

            % remove nans from legend array
            legends(cellfun(@(legends) any(isnan(legends)),legends)) = [];
            legends_complex = cellfun(@(x) x(~isnan(x)), legends_complex, 'UniformOutput', false);
            % Find empty columns
            emptyColumns = all(cellfun('isempty', legends_complex), 1);
            % Delete empty columns
            legends_complex = legends_complex(:, ~emptyColumns);

            if isempty(legends_list)
                legends_list = repmat(line(), 12, 1);
            end

            %labeling
            %pressure
            %if GUI || any(strcmp(varargin,'Scaled')) % Scaled
                as{7}.Title.String='Pressure Distribution Scaled';
                as{7}.XLabel.String='x [m]';
                % double y axis labels and legend:
                as_right{7}.YLabel.String='Im P [pa]';
                as{7}.YLabel.String='Re P [pa]';
                list=legends_list(5:6,:);
                list=list(:);
                legend(as{7},list(:),legends_complex{1:2,:},'color','w')
            %end
            %if GUI || any(strcmp(varargin,'Abs')) % Abs
                as{9}.Title.String='Pressure Distribution Abs/Phase';
                as{9}.XLabel.String='x [m]';
                % double y axis labels and legend:
                as_right{9}.YLabel.String='Phase P [Degrees]';
                as{9}.YLabel.String='Abs P [pa]';
                list=legends_list(7:8,:);
                list=list(:);
                legend(as{9},list(:),legends_complex{3:4,:},'color','w')
            %end
            %if GUI || (~any(strcmp(varargin,'Abs')) && ~any(strcmp(varargin,'Scaled'))) % normal
                as{1}.Title.String='Pressure Distribution';
                as{1}.XLabel.String='x [m]';
                as{1}.YLabel.String='P [pa]';
                legend(as{1},legends_complex{1:2,:})
            %end
            
            %velocity
            %if GUI || any(strcmp(varargin,'Scaled')) % Scaled
                as{8}.Title.String='Velocity Distribution Scaled';
                as{8}.XLabel.String='x [m]';
                % double y axis labels and legend:
                as_right{8}.YLabel.String='Im U [m^3 /s]';
                as{8}.YLabel.String='Re U [m^3 /s]';
                list=legends_list(9:10,:);
                list=list(:);
                legend(as{8},list(:),legends_complex{1:2,:},'color','w')
            %end

            %if GUI || any(strcmp(varargin,'Abs')) % Abs
                as{10}.Title.String='Velocity Distribution Abs/Phase';
                as{10}.XLabel.String='x [m]';
                % double y axis labels and legend:
                as_right{10}.YLabel.String='Phase U [Degrees]';
                as{10}.YLabel.String='Abs U [m^3 /s]';
                list=legends_list(11:12,:);
                list=list(:);
                legend(as{10},list(:),legends_complex{3:4,:},'color','w')
            %end

            %if GUI || (~any(strcmp(varargin,'Abs')) && ~any(strcmp(varargin,'Scaled'))) % normal
                as{2}.Title.String='Velocity Distribution';
                as{2}.XLabel.String='x [m]';
                as{2}.YLabel.String='U [m^3 /s]';
                legend(as{2},legends_complex{1:2,:})
            %end
            % temperature
                as{3}.Title.String='Temperature Distribution';
                legend(as{3},legends)
                as{3}.XLabel.String='x [m]';
                as{3}.YLabel.String=('T [k]');

            if GUI || any(strcmp(varargin,'Derived'))
                legend(as{4},legends)
                as{4}.XLabel.String='x [m]';
                ylabel(as{4},'$\dot{E}\quad[W] $','Interpreter','latex','FontWeight','bold')
                as{4}.Title.String='Acoustic Power Distribution';
                legend(as{5},legends)
                as{5}.XLabel.String='x [m]';
                ylabel(as{5},'$\dot{H}\quad[W] $','Interpreter','latex','FontWeight','bold')
                as{5}.Title.String='Total Power Distribution';
                legend(as{6},legends)
                as{6}.XLabel.String='x [m]';
                ylabel(as{6},'$\dot{m}\quad[kg/s] $','Interpreter','latex','FontWeight','bold')
                as{6}.Title.String='Mass Flux Distribution';
            end
            if GUI
                varargout={fs};
            else % showing only what was chosen (type of graph and derived)
                if any(strcmp(varargin,'Scaled'))
                    close(1)
                    close(2)
                    close(9)
                    close(10)
                    for i = 3:8
                        fs{i}.Visible='On';
                    end
                elseif any(strcmp(varargin,'Abs'))
                    close(1)
                    close(2)
                    close(7)
                    close(8)
                    for i = 3:10
                        if i ~= 7 && i~= 8
                            fs{i}.Visible='On';
                        end
                    end
                else % normal type
                    close(7)
                    close(8)
                    close(9)
                    close(10)
                    for i = 1:6
                        fs{i}.Visible='On';
                    end
                end

                if ~any(strcmp(varargin,'Derived'))
                    close(4)
                    close(5)
                    close(6)
                end
            end
        end
        %% copy system
        function cp=CopySystem(obj,newName,varargin)
            %this function copies a system, creating a new  identical
            %system with copies of all components. the name of the new
            %system will be defined by "new name".
            %cp=sys.CopySystem(newName) will copy components, branches and links
            %with a definition "copy of".
            %cp=sys.CopySystem(newName,newcomponentnames) will copy components
            % with  given names (as a cell array). similarily it can be run
            % with
            %cp=sys.CopySystem(newName,newcomponentnames,newbranchnames)
            % and
            %cp=sys.CopySystem(newName,newcomponentnames,newbranchnames,newlinknames)
            % this order of parameter must be kept, but empty variables can
            % be given if no name is specified: (example)
            %cp=sys.CopySystem(newName,newcomponentnames,{},newlinknames)
            cp = obj.copyElement;
            cp.Components_H={};
            cp.Links_H={};
            cp.Links = {};
            cp.Name=newName;
            
            clipboard_components=CopyGroup(obj,'Component',[1,length(obj.Components)],obj.Components);
            cp.Components_H=clipboard_components;
            for i=1:length(clipboard_components)
                cp.Components_H{i}.System_H=cp;
                cp.Components_H{i}.Location=i;
                cp.Components_H{i}.System=cp.Name;
                if isempty(varargin) || isempty(varargin{1}) %will be copied with the name "Copy of"
                    cp.Components_H{i}.rename(['copy of ',cp.Components_H{i}.Name]);
                else
                    cp.Components_H{i}.rename(varargin{1}{i});
                end

            end

            % Does changes to components that save references to other
            % objects in the system. Currently, this is just TA_Calculator,
            % but if there will be more of this behavior, it should be
            % addressed better.
            for i=1:length(cp.Components_H)
                if isa(cp.Components_H{i}, 'TA_Calculator')
                    cp.Components_H{i}.updateRelevantComponents(obj);
                end
            end

            for i=1:length(obj.Branches)
                if length(varargin)<2 || isempty(varargin{2})
                    bname=['Copy_of_',obj.Branches{i}];
                else
                    bname=obj.Branches_H{i}.CopySystem(varargin{2}{i});
                end
                cp.Branches_H{i}=obj.Branches_H{i}.CopySystem(bname);
                cp.Branches{i}=bname;
                cp.Branches_H{i}.Mother_System=newName;
                cp.Branches_H{i}.Mother_System_H=cp;
                cp.Branches_H{i}.Location=i;
            end

            cp.Links = obj.Links;
            if length(varargin)<3 || isempty(varargin{3}) %will be copied with the name "Copy of"
                clipboard_links=CopyGroup(obj,'Link',[1,length(obj.Links)]);
            else
                clipboard_links=CopyGroup(obj,'Link',[1,length(obj.Links)],varargin{3});
            end
            cp.Links_H=clipboard_links;
            for i=1:length(clipboard_links)
                cp.Links_H{i}.System_H=cp;
                cp.Links_H{i}.Location=i;
                cp.Links_H{i}.System=cp.Name;
                cp.Links_H{i}.rename(cp.Links_H{i}.Name);
                for j = 1:length(obj.Components)
                    [cp.Links_H{i}.Linked_Properties]=cp.SearchArray(...
                        cp.Links_H{i}.Linked_Properties,obj.Components{j},cp.Components{j});
                end
                % overwriting the listeners with new ones
                cp.Links_H{i}.Add_Link
            end

            if ~isempty(cp.Pressure) %if the original system was already run
                cp.runSystem(1);
            end
        end
        %% insert a component to a system
        % this function inserts a component into the system.
        % input is a handle to the component
        % Insert(component) puts the component at the end of the system
        % Insert(component,Loc) inserts the component into the location
        % specified by "loc"

        function Insert(obj,varargin)

            % asses and asign input
            if length(varargin)==2
                component=varargin{1};
                newLoc=varargin{2};
            elseif length(varargin)==1
                component=varargin{1};
                newLoc=length(obj.Components)+1;
            else
                error('wrong input format')
            end
            if ~isempty(component.System)
                error('component already has a system')
            end
            if newLoc>length(obj.Components)+1
                error('system has only %d components',length(obj.Components))
            end
            if newLoc<1||floor(newLoc)~=newLoc
                error('location must be a positve integer')
            end
            if any(strcmp(obj.Components,component.Name))
                error(['A component already exists with the name '...
                    component.Name])
            end
            %move components forward
            for i=length(obj.Components)+1:-1:newLoc+1
                obj.Components{i}=obj.Components{i-1};
                obj.Components_H{i}=obj.Components_H{i-1};
                obj.Components_H{i}.Location=i;
            end

            %add references between system and object
            obj.Components{newLoc}=component.Name;
            obj.Components_H{newLoc}=component;
            component.System=obj.Name;
            component.System_H=obj;
            component.Location=newLoc;
            obj.Recalc_GT;

            %adding default system properties to specific component properties, if they are not defined
            if strcmp(component.Inert,'null')
                component.Inert = component.System_H.Inert;
                component.Reactive = component.System_H.Reactive;
                component.Dry_Switch = component.System_H.Dry_Switch;
            end
        end
        %% remove a component from a system
        function Remove(obj,component)
            %this function removes a component from the system. the
            %input can be either a handle to the component or a string with
            %component name

            %find location of object in the system and convert from name to
            %actual component if neccesary
            if ischar(component)
                Loc=obj.findLoc(component);
                component=obj.Components_H{Loc};
            else
                Loc=obj.findLoc(component.Name);
            end

            %check if object has a guess or a target associated with it
            [~,isGuess]=obj.SearchArray(obj.Guesses,component.Name,[]);
            [~,isTarget]=obj.SearchArray(obj.Targets,component.Name,[]);
            if isGuess
                error(['remove component "',component.Name,'" from guess list before removing it from system.',...
                    'use the Function "Remove_Guess"'])
            end
            if isTarget
                error(['remove component "',component.Name,'" from target list before removing it from system.',...
                    'use the Function "Remove_Target"'])
            end

            %check if object has a link associated with it
            for i = 1:length(obj.Links)
                [~,isLink] = obj.SearchArray(obj.Links_H{i}.Linked_Properties,component.Name,[]);
                if isLink
                    error(['remove component "',component.Name,'" from a link before removing it from system.',...
                        ' use the Function "Remove_Link"'])
                end
            end

            %check if object is in a calculator

            %remove system references from component
            component.System='';
            component.System_H=TA_System.empty;
            component.Location=[];

            %rearange system component to disclude components
            for i=Loc:length(obj.Components)-1
                obj.Components{i}=obj.Components{i+1};
                obj.Components_H{i}=obj.Components_H{i+1};
                obj.Components_H{i}.Location=i;
            end
            obj.Components_H(end)=[];
            obj.Components(end)=[];

            %redefine guesses and targets
            obj.Recalc_GT;

        end
        
         %% remove a link from a system
         function Remove_Link(obj,link)
            %this function removes a link from the system. the
            %input can be either a handle to the link or a string with
            %link name

            %find location of object in the system and convert from name to
            %actual link if neccesary
            if ischar(link)
                Loc=obj.findLoc_Link(link);
                link=obj.Links_H{Loc};
            else
                Loc=obj.findLoc_Link(link.Name);
            end
            
            % delete the listeners of the link
            for i = 1:length(obj.Links_H{Loc}.Listeners)
                delete(obj.Links_H{Loc}.Listeners{i})
            end
            obj.Links_H{Loc}.Listeners = [];

            %remove system references from link
            link.System='';
            link.System_H=TA_System.empty;
            link.Location=[];

            %rearange system link to disclude links
            for i=Loc:length(obj.Links)-1
                obj.Links{i}=obj.Links{i+1};
                obj.Links_H{i}=obj.Links_H{i+1};
                obj.Links_H{i}.Location=i;
            end
            obj.Links_H(end)=[];
            obj.Links(end)=[]; 

         end
        %% removes all links in system
        function Remove_All_Links(obj)
            for i = length(obj.Links_H):-1:1
                Remove_Link(obj,obj.Links{i})
            end
        end
        %% relocate a system component
        function Relocate(obj,component,new_Loc)
            %this function relocates a component which is arleady inside the system.
            %the input can be either a handle to the component or a string with
            %component name, and the component's new location

            %find location of object in the system and convert from name to
            %actual component if neccesary
            if ischar(component)
                old_Loc=obj.findLoc(component);
                component=obj.Components_H{old_Loc};
            else
                old_Loc=obj.findLoc(component.Name);
            end

            % checking legal new location
            if new_Loc<1||floor(new_Loc)~=new_Loc
                error('location must be a positve integer')
            end
            
            if old_Loc < new_Loc
               %move components backwards
                for i = old_Loc+1:new_Loc
                    obj.Components{i-1}=obj.Components{i};
                    obj.Components_H{i-1}=obj.Components_H{i};
                    obj.Components_H{i-1}.Location=i-1;
                end

            elseif old_Loc > new_Loc
               %move components forward 
                for i = old_Loc:-1:new_Loc+1
                    obj.Components{i}=obj.Components{i-1};
                    obj.Components_H{i}=obj.Components_H{i-1};
                    obj.Components_H{i}.Location=i;
                end
            else
                warning('The component was moved to the same location')
            end

            % add references between system and object
            obj.Components{new_Loc}=component.Name;
            obj.Components_H{new_Loc}=component;
            component.Location=new_Loc;

            %redefine guesses and targets
            obj.Recalc_GT;
        end
        %% replace a system component
        function Replace(obj,old,new)
            %this function repalces a component from the system with a new one. the
            %input can be either a handle to the component or a string with
            %component name

            %find location of object in the system and convert from name to
            %actual component if neccesary
            if ischar(old)
                Loc=obj.findLoc(old);
                old=obj.Components_H{Loc};
            else
                Loc=obj.findLoc(old.Name);
            end

            %check if object has a guess or a target associated with it
            [~,isGuess]=obj.SearchArray(obj.Guesses,old.Name,[]);
            [~,isTarget]=obj.SearchArray(obj.Targets,old.Name,[]);
            if isGuess
                error(['remove component "',old.Name,'" from guess list before removing it from system',...
                    'use the Function "Remove_Guess"'])
            end
            if isTarget
                error(['remove component "',old.Name,'" from target list before removing it from system',...
                    'use the Function "Remove_Target"'])
            end

            %replace references
            old.System=[];
            old.Location=[];
            old.System_H.empty;

            new.System=obj.Name;
            new.System_H=obj;
            new.Location=Loc;

            obj.Components_H(Loc)=new;
            obj.Components(Loc)=new.Name;
        end
 
        %% remove Guess from system
        %this function removes the nth guess from the guess list
        function Remove_Guess(obj,n)
            if length(obj.Guesses)<n
                error('system has only %d guesses',length(obj.Guesses))
            end
            obj.Guesses(n)=[];
            obj.Guesses_RIAP(n)=[];
            obj.Guesses_ID(n)=[];
        end
        %% remove Target from system
        %this function removes the nth target from the target list
        function Remove_Target(obj,n)
            if length(obj.Targets)<n
                error('system has only %d targets',length(obj.Targets))
            end
            obj.Targets(n)=[];
            obj.Targets_RIAP(n)=[];
            obj.Targets_ID(n)=[];
        end
        %% empty variables
        function Empty_Variables(obj)
            obj.Pressure=[];
            obj.Velocity=[];
            obj.Temperature=[];
            obj.X=[];
            obj.Mass_Flux=[];
            obj.Total_Power=[];
            obj.Acoustic_Power=[];
            for i=1:length(obj.Components_H)
                obj.Components_H{i}.Empty_Varibles
            end
            for i=1:length(obj.Branches)
                obj.Branches_H{i}.Empty_Varibles
            end
        end
        %%
 function [T1,T11, C1, Tm_tot, Cm_tot, x_axis] = Calc_T1C1(obj, messageCallback)

    if nargin < 2 || isempty(messageCallback)
        messageCallback = @(msg, type) fprintf('%s: %s\n', upper(type), msg); % Default to console output
    end
    
    % Calculate oscillating temperature T1 and mass fraction C1
    % along with mean mass fraction Cm, temperature Tm, and x-axis.
    % Step 1: Extract Scaling Factors from System
    lambda = obj.Components_H{1}.Length; % Length of the first component
    omega = 2 * pi * obj.Frequency; % Angular frequency
    P_ref = obj.P_m; % Mean pressure of the system
    T_ref = obj.Begin(3); % Reference temperature (initial temperature from Begin)
    
    % Use Mixture_Properties for additional values
    [~, ~, ~, rho_ref, ~, ~, ~, M_mix_ref, ~, cp_ref, ~, ~, ~, ~, ~, ~] = ...
        Mixture_Properties(P_ref, T_ref, obj.Dry_Switch, obj.Mixture_Array);
    N0 = M_mix_ref * rho_ref; % Reference molar density
    
   % Debugging: Display extracted values
   % disp('Extracted Scaling Factors:');
   % disp(['lambda = ', num2str(lambda), ' m']);
   % disp(['omega = ', num2str(omega), ' rad/s']);
   % disp(['P_ref = ', num2str(P_ref), ' Pa']);
   % disp(['T_ref = ', num2str(T_ref), ' K']);
   % disp(['rho_ref = ', num2str(rho_ref), ' kg/m^3']);
   % disp(['cp_ref = ', num2str(cp_ref), ' J/(kg·K)']);
   % disp(['N0 = ', num2str(N0), ' mol/m^3']);
    
    % Step 2: Initialize Variables
    x_axis = [];
    for i = 1:length(obj.Components)
        if obj.Components_H{i}.Length > 0
            x_axis = [x_axis; obj.Components_H{i}.X];
        end
    end
    Cm_tot = [];
    Tm_tot = [];
    T1 = zeros(length(x_axis), 1);
    T11 = T1;
    C1 = zeros(length(x_axis), 1);
    start = 0;
  

    
    % Step 3: Loop Through Components to Calculate T1 and C1
    for c_i = 1:length(obj.Components)
      
        if obj.Components_H{c_i}.Length > 0
            [FD, Fv, Falpha] = obj.Components_H{c_i}.CalculateFs;
            A = obj.Components_H{c_i}.get_gas_area; % Cross-sectional area
            
            % Calculate mean mass fraction Cm and temperature Tm
            Cm = zeros(length(obj.Components_H{c_i}.X), 1);
            Tm = Cm;
            
            for j = 1:length(obj.Components_H{c_i}.X)
                [~, ~, ~, ~, ~, ~, ~, M_mix, ~, ~, ~, ~, ~, ~, Cm(j), ~] = ...
                    Mixture_Properties(obj.P_m, obj.Components_H{c_i}.Temperature(j), ...
                    obj.Components_H{c_i}.Dry_Switch, obj.Components_H{c_i}.Mixture_Array);
                    Tm(j) = obj.Components_H{c_i}.Temperature(j);
            end
            Cm_tot = [Cm_tot; Cm];
            Tm_tot = [Tm_tot; Tm];
            
            % Calculate derivatives
            x = obj.Components_H{c_i}.X;
            dCm = obj.diff12(Cm, x);
            dTm = obj.diff12(obj.Components_H{c_i}.Temperature, x);
            
            % Compute T1 and C1
            for j = 1:length(obj.Components_H{c_i}.X)
                [~, ~, ~, rho, Pr, Sc, ~, ~, ~, cp, ~, ~, ~, ~, ~] = ...
                    Mixture_Properties(obj.P_m, Tm(j), ...
                    obj.Components_H{c_i}.Dry_Switch, obj.Components_H{c_i}.Mixture_Array);
               

try
    % Fetch cross-sectional area
    A = obj.Components_H{c_i}.get_gas_area;

    % Debugging raw output
     %disp('Raw output of get_gas_area:');
     %disp(A);
    % disp(['Class of A: ', class(A)]);
    % disp(['Size of A: ', mat2str(size(A))]);
    % Process A to ensure scalar
    if isnumeric(A)
        % Reduce multidimensional numeric array to scalar
        A = mean(A(:));
    elseif iscell(A)
        % Convert cell array to numeric and reduce to scalar
        A = mean(cell2mat(A(:)));
    elseif isstruct(A)
        % Extract relevant field from structure (adjust 'someField')
        A = A.someField;
    else
        error('Unexpected type of A returned by get_gas_area');
    end

    % Ensure A is scalar
    assert(isscalar(A), 'Cross-sectional area A must be scalar');
    
    % Debugging A:
    % disp(['Final scalar value of A: ', num2str(A)]);

catch ME
    % Display error details
    disp(['Error with cross-sectional area A for Component = ', num2str(c_i)]);
    disp(['Error Message: ', ME.message]);
    rethrow(ME);
end
                 % Dimensional T1
                T11(start + j) = (Falpha(j) * obj.Components_H{c_i}.Pressure(j)) / (rho * cp) + ...
                    1i * ((Falpha(j) - Pr * Fv(j)) / (Fv(j) * (1 - Pr))) * ...
                     obj.Components_H{c_i}.Velocity(j) * dTm(j) / (omega * A);

                % Dimensional C1
                C1(start + j) = (-Cm(j) * (1 - FD(j)) * obj.Components_H{c_i}.Pressure(j) / P_ref + ...
                    1i * ((FD(j) - Sc * Fv(j)) / (Fv(j) * (1 - Sc))) * ...
                    obj.Components_H{c_i}.Velocity(j) * dCm(j) / (omega * A)) / M_mix;
                
            end
            start = start + length(obj.Components_H{c_i}.X);
        end
    end
    
    % Step 4: Sanity Checks for Physical Consistency
    if any(abs(T1) > Tm_tot)
        warning('Oscillating Temperature is unphysical, leads to negative Temperature');
    end
    if any(abs(C1) > Cm_tot)
        warning('Oscillating Mass Fraction is unphysical, leads to negative concentration');
    end
    if any(abs(obj.Pressure) > obj.P_m)
        warning('Oscillating pressure is unphysical, leads to negative pressure');
    end

 end


        %% mixture properties for ODE
        % mixture properties
        function  [ nu, alpha, D, rho, Pr, Sc, gamma, M, cp_mol,cp,k,sp,lh,Tref,CB,lh_m]...
                = Mixture_Properties_ODE(obj,p,T,dryswitch,Array)
            if p == obj.saved_p && T == obj.saved_T
                nu=obj.saved_m_properties(1); alpha=obj.saved_m_properties(2);
                D=obj.saved_m_properties(3); rho=obj.saved_m_properties(4);
                Pr=obj.saved_m_properties(5); Sc=obj.saved_m_properties(6);
                gamma=obj.saved_m_properties(7); M=obj.saved_m_properties(8);
                cp_mol=obj.saved_m_properties(9); cp=obj.saved_m_properties(10);
                k=obj.saved_m_properties(11); sp=obj.saved_m_properties(12);
                lh=obj.saved_m_properties(13); Tref=obj.saved_m_properties(14);
                CB=obj.saved_m_properties(15); lh_m=obj.saved_m_properties(16);
            else
                [ nu, alpha, D, rho, Pr, Sc, gamma, M, cp_mol,cp,k,sp,lh,Tref,CB,lh_m] = Mixture_Properties(p,T,dryswitch,Array);
                obj.saved_p=obj.P_m;
                obj.saved_T=T;
                obj.saved_m_properties=[ nu, alpha, D, rho, Pr, Sc, gamma, M, cp_mol,cp,k,sp,lh,Tref,CB,lh_m];
            end
        end
        %% assisting function for finding the location of a component
        function I=findLoc(obj,componentName)
            %this function finds the location of a component in the system
            %input is a string with component name
            I=0;
            for i=1:length(obj.Components)
                if strcmp(obj.Components(i),componentName)
                    I=i;
                end
            end
            if ~I
                error(['no component exists in system with the name ',componentName])
            end
        end
        %% assisting function for finding the location of a branch
        function I=findLoc_Branch(obj,branchName)
            %this function finds the location of a branch in the system
            %input is a string with branch name
            I=0;
            for i=1:length(obj.Branches)
                if strcmp(obj.Branches(i),branchName)
                    I=i;
                end
            end
            if ~I
                error(['no branch exists in system with the name ',branchName])
            end
        end
         %% assisting function for finding the location of a link
         function I=findLoc_Link(obj,linkName)
            %this function finds the location of a link in the system
            %input is a string with link name
            I=0;
            for i=1:length(obj.Links)
                if strcmp(obj.Links(i),linkName)
                    I=i;
                end
            end
            if ~I
                error(['no link exists in system with the name ',linkName])
            end
         end

        %% A function which validates added guess to possible linked parameter
        function Added_Guess_Link_Validation(obj,property_array)
            % a function which validates if a guess can be added to a parameter in link
            % if adding the guess is allowed, the function will let the adding guess code continue
            % if adding the guess is forbbiden, the function will stop the code with a message
            % property_array -> contains the parameter path array, by the fixed cases (e.g. {'Duct1','Length'})

            % First checking if the chosen parameter is in a link:
            for i = 1:length(obj.Links)
                 par_loc_in_link = obj.Comparearray(property_array,obj.Links_H{i}.Linked_Properties);

                 if par_loc_in_link ~= 0   % if parameter exists in a link, then validates by validation cases:
                     switch obj.Links_H{i}.Link_Type

                         case 'Equal' % only 1 guess is allowed in link
                             if ~isempty(obj.Guesses)
                                for j = 1:length(obj.Links_H{i}.Linked_Properties)
                                    if j ~= par_loc_in_link % skipping chosen parameter to allow it to be a guess multiple times
                                        guess_exist_in_link = obj.Comparearray(obj.Links_H{i}.Linked_Properties{j},obj.Guesses); % checking each parameter in link, if it is a guess
                                            if guess_exist_in_link
                                                error('Only one guess is allowed in a Link Equal type')
                                            end
                                    end
                                end
                             end
                            
                         case 'Relation' % no guess allowed in link
                             error('No guesses are allowed in a Link Relation type')

                         case {'Equal_Hierarchy','Relation_Hierarchy'} % only primary parameter can be guess
                             if par_loc_in_link ~= 1
                                 error('Only primary parameters can be be guesses in Link Hierarchy types')
                             end
                     end
                     
                 end
            end
        end

        %% assisting function for getting all the listeners of a specific property (in multiple links)
        function listeners_locations_array = get_property_listeners_locations(obj,property)
            listeners_locations_array = {};
            for i = 1:length(obj.Links_H)
                listeners_locations_array{end+1} = obj.Comparearray(property,obj.Links_H{i}.Linked_Properties);
            end
        end

        %% assisting function for getting all the listeners of a specific property (in multiple links)
        function turn_off_or_on_property_multi_listeners(obj,listeners_locations_array,action)
            if strcmp(action, "off")
                action = false;
            else
                action = true;
            end
            for i = 1:length(obj.Links_H)
                if listeners_locations_array{i} ~= 0
                    obj.Links_H{i}.Listeners{listeners_locations_array{i}}.Enabled = action;
                end
            end
        end

        %% assisting function for reassigning all guesses and targets
        function Recalc_GT(obj)
            
            %collect all guesses and targets
            Oldguess=obj.Guesses;
            Oldtarg=obj.Targets;
            Old_g_RIAP=obj.Guesses_RIAP;
            Old_t_RIAP=obj.Targets_RIAP;

            %empty all
            obj.Guesses={};
            obj.Targets={};
            obj.Guesses_ID={};
            obj.Targets_ID={};

            for i=1:length(Oldguess)
                obj.addGuess(Oldguess{i}{:})
            end
            for i=1:length(Oldtarg)
                obj.addTarget(Oldtarg{i}{:})
            end
            obj.Guesses_RIAP=Old_g_RIAP;
            obj.Targets_RIAP=Old_t_RIAP;
        end
        %% assisting function for collecting guess values from system
        function values=CollectGuessValues(obj)
            values=nan(1,length(obj.Guesses))+1i*nan(1,length(obj.Guesses));
            for i=1:length(obj.Guesses)
                switch obj.Guesses_ID{i}(1)
                    %different types of guesses, see in "addGuess"
                    case 1
                        values(i)=obj.(obj.Guesses{i}{1});
                    case 2
                        values(i)=obj.(obj.Guesses{i}{1})(obj.Guesses{i}{2});
                    case 3
                        values(i)=obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}{2});
                    case 4
                        values(i)= obj.Components_H{obj.Guesses_ID{i}(2)}.(obj.Guesses{i}...
                            {2})(obj.Guesses{i}{3});
                    case 5
                        values(i)= obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{3});
                    case 6
                       values(i)= obj.Branches_H{obj.Guesses{i}{1}}...
                            .Components_H{obj.Guesses_ID{i}(2)}...
                            .(obj.Guesses{i}{3})(obj.Guesses{i}{4});
                end
                switch obj.Guesses_RIAP{i}
                    case 'R'
                        values(i)=real(values(i));
                    case 'I'
                        values(i)=imag(values(i));
                    case 'A'
                        values(i)=abs(values(i));
                    case 'P'
                        values(i)=angle(values(i));
                end

            end
        end
        %% assisting function for collecting target values from system
        function values=CollectTargetValues(obj)
            values=nan(1,length(obj.Targets))+1i*nan(1,length(obj.Targets));
            for i=1:length(obj.Targets)
                switch obj.Targets_ID{i}(1)
                    %different types of targets, see in "addTarget"
                    case 1
                        values(i)=obj.(obj.Targets{i}{1});
                    case 2
                        values(i)=obj.(obj.Targets{i}{1})(obj.Targets{i}{2});
                    case 3
                        if ~isempty(obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2}))
                            values(i)=obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2});
                        end
                    case 4
                        if ~isempty(obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{2}))
                            values(i)= obj.Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}...
                                {2})(obj.Targets{i}{3});
                        end
                    case 5
                        if ~isempty(obj.Branches_H{obj.Targets{i}{1}}...
                                .Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{3}))
                            values(i)= obj.Branches_H{obj.Targets{i}{1}}...
                                .Components_H{obj.Targets_ID{i}(2)}...
                                .(obj.Targets{i}{3});
                        end
                    case 6
                        if ~isempty(obj.Branches_H{obj.Targets{i}{1}}...
                                .Components_H{obj.Targets_ID{i}(2)}.(obj.Targets{i}{3}))
                            values(i)= obj.Branches_H{obj.Targets{i}{1}}...
                                .Components_H{obj.Targets_ID{i}(2)}...
                                .(obj.Targets{i}{3})(obj.Targets{i}{4});
                        end
                end
                switch obj.Targets_RIAP{i}
                    case 'R'
                        values(i)=real(values(i));
                    case 'I'
                        values(i)=imag(values(i));
                    case 'A'
                        values(i)=abs(values(i));
                    case 'P'
                        values(i)=angle(values(i));
                end

            end

        end
        %% Assisting function for changing system property
        % this function changes a system property as specified by an
        % array

        % Change_Property({property},value)- input is a string with a property name
        % e.g Change_Property({'P_m'}) changes P_m.

        % Change_Property({property,i},value) input is a string with property name
        % and an integer index
        % e.g Change_Property({'Begin',3}) changes Begin(3) (temperature at the
        % begining)

        % Change_Property({component,component_property},value) input is a string with
        % component name and another one with the component property
        % name.
        % e.g Change_Property ({'Duct1','length'},value)
        % changes the length of component Duct1

        % Change_Property({component,component_property,index},value) input is a string with
        % component name, another one with the component property
        % name, and an index.
        % e.g Change_Property ({'Duct1','Pressure',1},value) changes the pressure of
        % at the begining of Duct1
        
        % Change_Property({branchnum,component,component_property},value)
        % input is a branch number, a string with component name, another
        % one with the component proper name, and an index.
        % e.g Change_Property ({1,'Duct1','Length'},value) changes the Length of
        % Duct1 in branch 1
        
          % Change_Property({branchnum,component,component_property,index},value)
        % input is a branch number, a string with component name, another
        % one with the component proper name, and an index.
        % e.g Change_Property ({1,'Duct1','Pressure',1},value) changes the pressure of
        % at the begining of Duct1 in branch 1

        function Change_Property(obj,info_array,value)
            % case 1
            if length(info_array)==1&&ischar(info_array{1})
                obj.(info_array{1})=value;
                %case 2
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&isnumeric(info_array{2})
                obj.(info_array{1})(info_array{2})=value;
                %case 3
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&ischar(info_array{2})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.(info_array{2})=value;
                %case 4 
            elseif length(info_array)==3&&ischar(info_array{1})&&...
                    ischar(info_array{2})&&isnumeric(info_array{3})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.(info_array{2})(info_array{3})=value;
                %case 5
            elseif length(info_array)==3&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                obj.Branches_H{info_array{1}}.Components_H{I}.(info_array{3})=value;
                %case 6
            elseif length(info_array)==4&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})&&isnumeric(info_array{4})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                obj.Branches_H{info_array{1}}.Components_H{I}.(info_array{3})(info_array{4})=value;
            else
                error('wrong format for changing a property')
            end
        end
        %% Assisting function for evaluating system property
        % this function evaluates a system property as specified by an
        % array

        % case 1:
        % Evaluate_Property({property})- input is a string with a property name
        % e.g Evaluate_Property({'P_m'}) evaluates P_m.

        % case 2:
        % Evaluate_Property({property,i}) input is a string with property name
        % and an integer index
        % e.g Evaluate_Property({'Begin',3}) evaluates Begin(3) (temperature at the
        % begining)

        % case 3:
        % Evaluate_Property({component,component_property}) input is a
        % component name and another one with the component property
        % name. the component name can also be a component object (case
        % 3.5).
        % e.g Evaluate_Property ({'Duct1','length'})
        %     Evaluate_Property ({Duct1,'length'})
        % evaluates the length of component Duct1

        % Evaluate_Property(component,component_property,i) input is a 
        % component name, another one with the component property
        % name, and an index. the component name can also be a component 
        % object (case 4.5). index can be "end".
        % e.g Evaluate_Property ({'Duct1','Pressure',1}) 
        %     Evaluate_Property ({Duct1,'Pressure',1}) 
        % evaluates the pressure of
        % at the begining of Duct1

        function value=Evaluate_Property(obj,info_array)
            % case 1
            if length(info_array)==1&&ischar(info_array{1})    
                try
                    obj.(info_array{1});
                catch
                    error(['no property exists with the name ',info_array{1}])
                end
                value=obj.(info_array{1});
                %case 2
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&isnumeric(info_array{2})
                value=obj.(info_array{1})(info_array{2});
                %case 3
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&ischar(info_array{2})
                I=obj.findLoc(info_array{1});
                value=obj.Components_H{I}.(info_array{2});
                %case 3.5
            elseif length(info_array)==2&&isa(info_array{1}, 'TA_Component')...
                    &&ischar(info_array{2})
                value=info_array{1}.(info_array{2});
                %case 4
            elseif length(info_array)==3&&ischar(info_array{1})&&...
                    ischar(info_array{2})&&(isnumeric(info_array{3}) ||...
                    ischar(info_array{3}) && strcmp(info_array{3}, "end"))
                I=obj.findLoc(info_array{1});
                if ischar(info_array{3}) && strcmp(info_array{3}, "end")
                    value=obj.Components_H{I}.(info_array{2})(end);
                else
                    value=obj.Components_H{I}.(info_array{2})(info_array{3});
                end
                %case 4.5
            elseif length(info_array)==3&&isa(info_array{1}, 'TA_Component')&&...
                    ischar(info_array{2})&&(isnumeric(info_array{3}) ||...
                    ischar(info_array{3}) && strcmp(info_array{3}, "end"))
                if ischar(info_array{3}) && strcmp(info_array{3}, "end")
                    value=info_array{1}.(info_array{2})(end);
                else
                    value=info_array{1}.(info_array{2})(info_array{3});
                end
                %case 5
            elseif length(info_array)==3&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                value=obj.Branches_H{info_array{1}}.Components_H{I}.(info_array{3});
                %case 6
            elseif length(info_array)==4&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})&&isnumeric(info_array{4})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                value=obj.Branches_H{info_array{1}}.Components_H{I}.(info_array{3})(info_array{4});
            else
                error('wrong format for evaluating a property')
            end
        end

        %% Assisting function for applying link at creation for protected components
        % this function changes a system property as specified by an
        % array.
        % it is different from the normal Change_Property to bypass 'protected' parameters

        % Change_Property_Link({property},value)- input is a string with a property name
        % e.g Change_Property({'P_m'}) changes P_m.

        % Change_Property_Link({property,i},value) input is a string with property name
        % and an integer index
        % e.g Change_Property({'Begin',3}) changes Begin(3) (temperature at the
        % begining)

        % Change_Property_Link({component,component_property},value) input is a string with
        % component name and another one with the component property
        % name.
        % e.g Change_Property_Link ({'Duct1','length'},value)
        % changes the length of component Duct1

        % Change_Property_Link({component,component_property,index},value) input is a string with
        % component name, another one with the component property
        % name, and an index.
        % e.g Change_Property ({'Duct1','Pressure',1},value) changes the pressure of
        % at the begining of Duct1
        
        % Change_Property_Link({branchnum,component,component_property},value)
        % input is a branch number, a string with component name, another
        % one with the component proper name, and an index.
        % e.g Change_Property ({1,'Duct1','Length'},value) changes the Length of
        % Duct1 in branch 1
        
        % Change_Property_Link({branchnum,component,component_property,index},value)
        % input is a branch number, a string with component name, another
        % one with the component proper name, and an index.
        % e.g Change_Property ({1,'Duct1','Pressure',1},value) changes the pressure of
        % at the begining of Duct1 in branch 1

        function Change_Property_Link(obj,info_array,value)
               % case 1
            if length(info_array)==1&&ischar(info_array{1})
                obj.(info_array{1})=value;
                % case 2
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&isnumeric(info_array{2})
                obj.(info_array{1})(info_array{2})=value;
                % case 3
            elseif length(info_array)==2&&ischar(info_array{1})...
                    &&ischar(info_array{2})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.Change_Par_Value_Through_TA_Component(info_array,3,value);
                % case 4 
            elseif length(info_array)==3&&ischar(info_array{1})&&...
                    ischar(info_array{2})&&isnumeric(info_array{3})
                I=obj.findLoc(info_array{1});
                obj.Components_H{I}.Change_Par_Value_Through_TA_Component(info_array,4,value);
                % case 5
            elseif length(info_array)==3&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                obj.Branches_H{info_array{1}}.Components_H{I}.Change_Par_Value_Through_TA_Component(info_array,5,value);
                % case 6
            elseif length(info_array)==4&&isnumeric(info_array{1})&&...
                    ischar(info_array{2})&&ischar(info_array{3})&&isnumeric(info_array{4})
                I=obj.Branches_H{info_array{1}}.findLoc(info_array{2});
                obj.Branches_H{info_array{1}}.Components_H{I}.Change_Par_Value_Through_TA_Component(info_array,6,value);
            else
                error('wrong format for changing a property')
            end
        end
        %% property meta data obtaining
        % a function that gets the component meta data, and the property's location in it.
        % metadata -> the meta data of the component of the specific property
        % metaloc -> the location of the property in the meta data of the component
        % par_array -> the array of the chosen property
    
        function [metadata,metaloc] = get_property_meta(obj,par_array)  
    
            % in case it is a component parameter with branch: (cases 5-6)
            if isnumeric(par_array{1})  % examples: {1,'Duct1','Length'} or {1,'Duct1','Pressure',1}
                I = obj.findLoc(par_array{2}); % finding location of the component in the system in order to reach it's handle
                metadata = metaclass(obj.Components_H{I}); % creating a meta class to the component in order to reach the parameter's info
                for i = 1:length(metadata.PropertyList) % searching for the location of the parameter
                    if strcmp(metadata.PropertyList(i).Name,par_array{3})
                        metaloc = i;
                        break
                    end
                end
    
            else 
                % in case it is a component parameter without branch: (cases 3-4)
                if length(par_array) >= 2 && ischar(par_array{2})  % examples: {'Duct1','Length'} or {'Duct1','Pressure',1}
                    I = obj.findLoc(par_array{1}); % finding location of the component in the system in order to reach it's handle
                    metadata = metaclass(obj.Components_H{I}); % creating a meta class to the component in order to reach the parameter's info
                    for i = 1:length(metadata.PropertyList) % searching for the location of the parameter
                        if strcmp(metadata.PropertyList(i).Name,par_array{2})
                            metaloc = i;
                            break
                        end
                    end
    
                % in case it is a system parameter: (cases 1-2)
                else   % examples: {'P_m'} or {'Begin',3}
                    metadata = metaclass(obj); % creating a meta class to the system in order to reach the parameter's info
                    for i = 1:length(metadata.PropertyList) % searching for the location of the parameter
                        if strcmp(metadata.PropertyList(i).Name,par_array{1})
                            metaloc = i;
                            break
                        end
                    end  
                end 
            end
        end
    end
    methods(Static)
        %% assisting function for changing a guessed value
        %old is the old complex number
        %value is the new value
        %RIAP is the part to be changed
        function  new=changeguess(old,value,RIAP)
            switch RIAP
                case 'R'
                    %real value
                    new=value+1i*imag(old);
                case 'I'
                    %imaginary value
                    new=1i*value+real(old);
                case 'A'
                    %absolute value
                    [a1,a2]=pol2cart(angle(old),value);
                    new=a1+1i*a2;
                case 'P'
                    %phase
                    [a1,a2]=pol2cart(value,abs(old));
                    new=a1+1i*a2;
            end
        end
        %% assisting function for checking and manipulating the guess, targets and links array
        function [I_Array,value,indexes_in_array]=SearchArray(I_Array,Old,New)
            %this function searches the array defined by I_Array based on the
            % Expression in "Old". if New is empty it will simply return true
            % if it found the expression and false otherwise. if new is a string it will
            % replace the old string with the new
            %usually a guess or target array, or a link.
            % "indexes_in_array" will contain the indexes of "Old" in "I_Array"
            value=false;
            indexes_in_array = [];
            if isempty(New)
                flag=0;
            else
                flag=1;
            end
            for i=1:length(I_Array)
               % checking for a component, without a branch
                if length(I_Array{i})>1 && ischar(I_Array{i}{1}) && ischar(I_Array{i}{2})
                    if strcmp(I_Array{i}{1},Old)
                        value=true;
                        indexes_in_array = [indexes_in_array,i];
                        if flag
                            I_Array{i}{1}=New;
                        % else
                        %     return
                        end
                    end
               % checking for a component, with a branch
                elseif length(I_Array{i})>2 && isnumeric(I_Array{i}{1})...
                              && ischar(I_Array{i}{2}) && ischar(I_Array{i}{3})
                    if strcmp(I_Array{i}{2},Old)
                        value=true;
                        indexes_in_array = [indexes_in_array,i];
                        if flag
                            I_Array{i}{2}=New;
                        % else
                        %     return
                        end
                    end
                end
            end
        end

        %% Assisting function for comparing one array with an array of other arrays
        function answer=Comparearray(A1,A2)
            %this function compares the small array A1 with the large
            %array A2, comparing A1 to every value of A2. returns the
            %location of A1 in A2 or 0 if no such location exists.
            fun=@(a,b) length(a)==length(b)&&max(cellfun(@isequal, {a},{b}));
            answer=find(cellfun(@(c) fun(A1,c),A2));
            if isempty(answer)
                answer=0;
            end
        end
        %% assisting function for calculating derivative of a vector
        function [deriv] = diff12(u,x)
            %calculates first derivative of a function with 2d order accuracy
            %based on central differences for internal points and forward/bacward
            %differences for external points
            if length(x)>2 && length (u)>2
            xd = diff([x(3);x;x(end-2)]);  % <-- Corrected
            ud = diff([u(3);u;u(end-2)]);  % <-- Corrected
            deriv = (ud(1:end-1)./xd(1:end-1).*xd(2:end) ...
                + ud(2:end)./xd(2:end).*xd(1:end-1)) ...
                ./ (xd(2:end)+xd(1:end-1));
            else

            deriv = zeros(length(x));

            end
        end
       
    end

end

