classdef TA_Round_TBT <TA_Stack
    %Describes a round thermal buffer tube
    
    %% dependent geometrical properties    
    properties (SetObservable=true,Dependent)
        Diameter...                            % HX diameter          [m]
            (1,1) double{mustBeNonnegative}
        Area                                   % HX area              [m^2]
        Rh  (1,1) double ...
            {mustBeGreaterThan(Rh,1e-10),mustBeLessThan(Rh,1)}    % hydraulic radius     [m]
        Wall_Thickness...                      % thickness of wall    [m]
            (1,1) double{mustBeNonnegative}
        A_Wall...                              % Areacovered by solid [m^2]
            (1,1) double{mustBeNonnegative}
    end
    %%
    properties (Hidden)
        A_Solid=0 % the solid area inside the tube itself is 0

        % parameters which keep values and connects to the other parameters:
        Diameter_val ...                         % HX diameter          [m]
            (1,1) double{mustBeNonnegative}=0.1
        Wall_Thickness_val...                      % thickness of wall    [m]
            (1,1) double{mustBeNonnegative}=0
        A_Wall_val...                              % Areacovered by solid [m^2]
            (1,1) double{mustBeNonnegative}=0
    end
    
    methods
        %% constructor
        function obj = TA_Round_TBT(name,system,length,Diameter,solid,Wall_thickness,varargin)
            obj@TA_Stack(name,system,length,solid,varargin{:})
            obj.Diameter_val=Diameter;
            obj.A_Wall_val=pi*Diameter*Wall_thickness;
        end
        
         %% set method for diameter
        function set.Diameter(obj,value)
            obj.Diameter_val=value;
            obj.A_Wall_val=pi*obj.Diameter_val*obj.Wall_Thickness_val;
        end
        %% get method for diameter
        function value=get.Diameter(obj)
           value=obj.Diameter_val;
        end
        
        %% set method for area
        function set.Area(obj,value)
            obj.Diameter_val=(value/pi*4)^0.5;
            obj.A_Wall_val=pi*obj.Diameter_val*obj.Wall_Thickness_val;
        end
        %% get method for area
        function value=get.Area(obj)
           value=pi*obj.Diameter_val^2/4;
        end

        %% set method for Rh
        function  set.Rh(obj,value)
            obj.Diameter_val=value*4;
            obj.A_Wall_val=pi*obj.Diameter_val*obj.Wall_Thickness_val;
        end
        %% get method for Rh
        function value=get.Rh(obj)
            value=obj.Diameter_val/4;
        end

        %% set method for A_Wall
        function  set.A_Wall(obj,value)
            if value>=(pi*obj.Diameter_val^2/4)/2
                error('walls too thick')
            end
            obj.A_Wall_val=value;
            obj.Wall_Thickness_val=value/pi/obj.Diameter_val;
        end
        %% get method for A_Wall
        function value=get.A_Wall(obj)
            value=obj.A_Wall_val;
        end

        %% set method for wall thickness
        function  set.Wall_Thickness(obj,value)
            if value>=1
                error('solid area must be smaller than total area')
            end
            obj.Wall_Thickness_val=value;
            obj.A_Wall_val=value*pi*obj.Diameter_val;
        end
        %% get method for wall thickness
        function value=get.Wall_Thickness(obj)
            value=obj.Wall_Thickness_val;
        end

        %% geometrical functions
        %this function  assists in averaging a property based
        %on the thermal penetration depth and geometry
        %source:delta E.C version 6.4b2.7 user's guide, equations
        %10.59-10.60
        
        function  f=f_function(obj,delta)
            rh=obj.Diameter_val/4;
            if rh/delta<12.5              %small enough for bessel
                param=(1i-1).*2*rh./delta; %parameter for convenicence
                
                f=2*besselj(1,param)./(besselj(0,param).*param);
                
            elseif rh/delta>15            %boundary layer
                f=(1-1i)*delta/2/rh;
            else                            %intermediate-interpolation
                param=(1i-1).*2*rh./delta;
                f1=2*besselj(1,param)./(besselj(0,param).*param);
                f2=(1-1i)*delta/2/rh;
                f=f1+(rh-12.5)/(15-12.5)*(f2-f1);
            end
        end
        function epsilon=eps_function(obj,~,deltas,~)
            lll=obj.A_Wall_val/pi/obj.Diameter_val;
            epsilon=1/tanh((1+1i)*lll/deltas);
        end
    end
end