classdef TA_Plate_HX <TA_HX
    %Describes a parralel plate HX

    %% independent properties
    properties(SetObservable=true)
        Y0 (1,1) double{mustBeNonnegative}    % half the plate spacing  [m]
        Lp (1,1) double{mustBeNonnegative}    % half thickness of solid [m]      
    end

    properties (Hidden) % independent properties
        % parameters which keep values and connects to the other parameters:
        Diameter_val(1,1) double{mustBeNonnegative}  % HX diameter         [m] 
        Area_val=0.1
        A_Solid_val=0.1
        Gas_Area_Ratio_val (1,1) double{mustBeGreaterThan(Gas_Area_Ratio_val,0)...
            ,mustBeLessThan(Gas_Area_Ratio_val,1)} = 0.1
    end

    %% dependent geometrical properties
    properties (SetObservable=true,Dependent)
        Diameter (1,1) double{mustBeNonnegative}  % HX diameter         [m]
        Area            % HX area [m^2]         [m]
        A_Solid         % Area covered by solid    [m]
        Gas_Area_Ratio ...  % percentege of area covered by gas
            (1,1) double{mustBeGreaterThan(Gas_Area_Ratio,0)...
            ,mustBeLessThan(Gas_Area_Ratio,1.0001)} 
    end
    
    methods
        %% constructor
        function obj = TA_Plate_HX(name,system,length,D,y0,phi,solid,Q,varargin)
            obj@TA_HX(name,system,length,solid,Q,varargin{:})
            obj.Diameter_val=D;
            obj.Y0=y0;
            obj.Gas_Area_Ratio_val=phi;
            obj.Lp=y0*(1/phi-1);
            obj.Area_val=pi*D^2/4;
            obj.A_Solid_val=pi*D^2/4*(1-obj.Gas_Area_Ratio_val);
        end

        
         %% set method for diameter
        function set.Diameter(obj,value)
            obj.Diameter_val=value;
            obj.Area_val=pi*value^2/4;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*obj.Area_val;
        end
        %% get method for diameter
        function value=get.Diameter(obj)
            value=obj.Diameter_val;
        end

        %% set method for area
        function set.Area(obj,value)
            obj.Area_val=value;
            obj.Diameter_val=(value/pi*4)^0.5;
            obj.A_Solid_val=(1-obj.Gas_Area_Ratio_val)*value;
        end
        %% get method for area
        function value=get.Area(obj)
             value=obj.Area_val;
        end
        
        %% set method for A_solid
        function  set.A_Solid(obj,value)
            if value>=obj.Area_val
                error('solid area must be smaller than total area')
            end
            obj.A_Solid_val=value;
            obj.Gas_Area_Ratio_val=(obj.Area_val-value)/obj.Area_val;
        end
        %% get method for A_Solid
        function value=get.A_Solid(obj)
             value=obj.A_Solid_val;
        end

        %% set method for Gas_Area ratio
        function  set.Gas_Area_Ratio(obj,value)
            if value>=1
                error('solid area must be smaller than total area')
            end
            obj.Gas_Area_Ratio_val=value;
            obj.A_Solid_val=(1-value)*obj.Area_val;
        end
        %% get method for Gas_Area_Ratio
        function value=get.Gas_Area_Ratio(obj)
             value=obj.Gas_Area_Ratio_val;
        end
        
        %% geometrical functions
        %this function  assists in averaging a property based
        %on the thermal penetration depth and geometry
        %source:delta E.C version 6.4b2.7 user's guide, equations
        %10.59-10.60

        function  f=f_function(obj,delta)
        rh=obj.Y0;
        f=tanh((1+1i)*rh/delta)/((1+1i)*rh/delta);
        end
        function epsilon=eps_function(obj,deltak,deltas,~)
            rh=obj.Y0;
            lp=obj.Lp;
            epsilon=tanh((1+1i)*rh/deltak)/tanh((1+1i)*lp/deltas);
        end
        %% solid temperature calculation
        function T_solid=Calculate_Solid_Temperature(obj)
            % this function calculates the solid temperature based on 
            % DEC-delta E.C version 6.4b2.7 user's guide, section 10.7.1
            
            %collect properties from object and system
            p_m=obj.System_H.P_m;
            T_m=obj.Temperature(end);
            rh=obj.Y0;
            Array=obj.Mixture_Array;
            dryswitch=obj.Dry_Switch;

            % if strcmp(obj.Inert,"null") % choosing whether to take Array from system or component
            %     Array=obj.System_H.Mixture_Array;
            % else
            %     Array=obj.Mixture_Array;
            % end
            % if strcmp(obj.Dry_Switch,"null") % choosing whether to take Dry Switch from system or component
            %     dryswitch=obj.System_H.Dry_Switch;
            % else
            %     dryswitch=obj.Dry_Switch;
            % end

            omega=obj.System_H.Frequency*2*pi;

            %calculate properties
            [ ~, alpha, ~, ~, ~, ~, ~, ~, ~,~,k] = Mixture_Properties(p_m,T_m,dryswitch,Array);
            
            zeta= abs(obj.Velocity(end)/(obj.Area_val-obj.A_Solid_val))/omega;              % acoustic displacement
            xeff=min(2*zeta,obj.Length);
            
            deltak=(2*alpha/omega)^0.5;
            yeff=min(deltak,rh);                                                %effective distance from the plate
            h=k/yeff;                                                           %approximate convection coefficient
            T_solid=T_m+obj.Heat_Input/(h*(obj.Area_val-obj.A_Solid_val)*xeff/rh);      %DEC eq 10.113
        end
    end
end

