classdef TA_Link<matlab.mixin.Copyable
    %this class defines a link between two or more properties in a TA_System

    properties (SetAccess=?TA_System)
        System (1,:) char % Name of the "mother" system
        Name   (1,:) char % Name of the link
        Location double{mustBeInteger} % the location of the link in the mother system

        forGUI (1,1) logical = 0 % A flag for GUI interactions (equals 1 if it is for GUI)
    end
    properties (SetAccess=?TA_System)
        System_H TA_System                            % Handle to TA_System
    end
    properties 
        Linked_Properties cell = {}                   % A list of the properties names in string cells
        Linked_Properties_Values (1,:) double         % A list of the properties values, which changes accordingly
        Linked_Properties_Values_Revert (1,:) double  % A list of the properties values, kept to revert the changed values in case of an error
        Link_Fail_Flag (1,1) double = 0               % turns to the value 1 when a link fails. for now only used at applying the link at creation
        Link_Fail_Chain (1,1) double = 0              % turns to the value 1 when a link fails. used to revert beck values of chain of links
        Listeners (1,:) cell                          % Handle to the listener functions
        Properties_ID (1,:) cell                           % Symbolize different parameter cases, as described in Add_Link function below
    end

    properties (Abstract)
        Link_Type (1,:) char  % Type of the link
    end

    properties (AbortSet=true)
        Check_Abort
    end

methods
    %% constructor
    function obj = TA_Link(name,system,Linked_Properties,forGUI)
       % basic properties:
        obj.Name = name;
        obj.System = system.Name;
        obj.System_H = system;
        obj.forGUI = forGUI;
        obj.Linked_Properties = Linked_Properties;
        
        if length(Linked_Properties) < 2
            obj.Link_Fail_Flag = 1; % flags link fail
            error('Please choose at least 2 parameters to link')
        end

        for i = 1:length(obj.Linked_Properties)
            [metadata,metaloc] = get_property_meta(obj,i);

            % first general link validation as the following:
            % if one of the linked properties is not numeric or 'mustBeInteger', it is forbbiden
            if isempty(metadata.PropertyList(metaloc,1).Validation) || isempty(metadata.PropertyList(metaloc,1).Validation.Class) || ...
               ~strcmp(metadata.PropertyList(metaloc,1).Validation.Class.Name,'double') ||... % Checking if not numeric
               (~isempty(metadata.PropertyList(metaloc,1).Validation.ValidatorFunctions) && strcmp(func2str(metadata.PropertyList(metaloc,1).Validation.ValidatorFunctions{1}),'mustBeInteger')) % specific check for denying N_Sol_Points

                obj.Link_Fail_Flag = 1; % flags link fail
                error(['The parameter',' "', metadata.PropertyList(metaloc,1).Name, '" ', 'can not be linked'])
            end

            % in case of target parameter that is empty before the system was ran:
            if ((isa(metadata.PropertyList(metaloc).SetAccess, 'char') && strcmp(metadata.PropertyList(metaloc).SetAccess,'protected')) ||...
                        isa(metadata.PropertyList(metaloc).SetAccess, 'cell')) && isempty(obj.System_H.Pressure)
                error(['Run the system before linking the parameter',' "', metadata.PropertyList(metaloc,1).Name, '"'])
            end
            
            % if link is allowed, continuing with checking the parameters value and recording them:
            obj.Linked_Properties_Values(i) = obj.System_H.Evaluate_Property(obj.Linked_Properties{i});
            if obj.forGUI && ~isnumeric(obj.Linked_Properties{i}{end}) % adds the number 1 to first vector index of the parameter if it is for GUI and is scalar - out of convenience
                obj.Linked_Properties{i}{end+1} = 1;
            end

        end
        obj.Linked_Properties_Values_Revert = obj.Linked_Properties_Values;
        obj.Location=size(system.Links,2)+1;
        system.addLink(name);
        system.Links_H=[system.Links_H,{obj}];
    end
    
    %% adding link function
    function Add_Link(obj)

        for i = 1:size(obj.Linked_Properties,2)

                %case 1 - {'P_m'}      (in GUI scalar system parameters will be assigned as case 2 with index of 1)
            if length( obj.Linked_Properties{i})==1&&ischar( obj.Linked_Properties{i}{1})
                obj.Listeners{i} = addlistener(obj.System_H,...
                     obj.Linked_Properties{i}{1},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 1;
                
                %case 2 - {'Begin',3}
            elseif length( obj.Linked_Properties{i})==2&&ischar( obj.Linked_Properties{i}{1})...
                    &&isnumeric( obj.Linked_Properties{i}{2})
                obj.Listeners{i} = addlistener(obj.System_H,...
                     obj.Linked_Properties{i}{1},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 2;

                %case 3 - {'Duct1','Length'}    (in GUI scalar parameters will be assigned as case 4 with index of 1)
            elseif length( obj.Linked_Properties{i})==2&&ischar( obj.Linked_Properties{i}{1})...
                    &&ischar( obj.Linked_Properties{i}{2})
                I=obj.System_H.findLoc( obj.Linked_Properties{i}{1});
                obj.Listeners{i} = addlistener(obj.System_H.Components_H{I},...
                     obj.Linked_Properties{i}{2},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 3;

                %case 4 - {'Duct1','Pressure',1}
            elseif length( obj.Linked_Properties{i})==3&&ischar( obj.Linked_Properties{i}{1})&&...
                    ischar( obj.Linked_Properties{i}{2})&&isnumeric( obj.Linked_Properties{i}{3})
                I=obj.System_H.findLoc( obj.Linked_Properties{i}{1});
                obj.Listeners{i} = addlistener(obj.System_H.Components_H{I},...
                     obj.Linked_Properties{i}{2},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 4;

                %case 5 - {1,'Duct1','Length'}      (in GUI scalar branched parameters will be assigned as case 6 with index of 1)
            elseif length( obj.Linked_Properties{i})==3&&isnumeric( obj.Linked_Properties{i}{1})&&...
                    ischar( obj.Linked_Properties{i}{2})&&ischar( obj.Linked_Properties{i}{3})
                I=obj.System_H.Branches_H{ obj.Linked_Properties{i}{1}}.findLoc( obj.Linked_Properties{i}{2});
                obj.Listeners{i} = addlistener(obj.System_H.Branches_H{ obj.Linked_Properties{i}{1}}.Components_H{I},...
                     obj.Linked_Properties{i}{3},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 5;

                %case 6 - {1,'Duct1','Pressure',1}
            elseif length( obj.Linked_Properties{i})==4&&isnumeric( obj.Linked_Properties{i}{1})&&...
                    ischar( obj.Linked_Properties{i}{2})&&ischar( obj.Linked_Properties{i}{3})&&...
                    isnumeric( obj.Linked_Properties{i}{4})
                I=obj.System_H.Branches_H{ obj.Linked_Properties{i}{1}}.findLoc( obj.Linked_Properties{i}{2});
                obj.Listeners{i} = addlistener(obj.System_H.Branches_H{obj.Linked_Properties{i}{1}}.Components_H{I},...
                     obj.Linked_Properties{i}{3},'PostSet',@(source,data)obj.try_to_link(source,data,i));
                obj.Properties_ID{i} = 6;

            else
                error('wrong format for a property')
            end

        end
        
        try
            % applying the link at creation
            obj.Link_Fail_Flag = 0; % resets flag link fail at the attempt to execute a link.
            obj.System_H.Change_Property_Link(obj.Linked_Properties{1},obj.Linked_Properties_Values(1))
            if obj.Link_Fail_Flag == 1
                error(['Creation of the link is not possible with the current linked values, as specified in the error above.',...
                       char(10),'The link has been deleted'])
            end

            % checking if the link has a target parameter, and if the creation is in code and not GUI, then present a warning
            % which says that target parameters are not fully support yet, because their combination with a link is not fully
            % understood yet:
            if ~obj.forGUI
                [metadata,metaloc] = get_property_meta(obj,1);  % only the first parameter can be a target
                if (isa(metadata.PropertyList(metaloc).SetAccess, 'char') && strcmp(metadata.PropertyList(metaloc).SetAccess,'protected')) ||...
                        isa(metadata.PropertyList(metaloc).SetAccess, 'cell')
                    warning('Linking a system calculated parameter is not fully supported yet')
                end
            end

        catch me
            for i = 1:length(obj.Listeners)
                delete(obj.Listeners{i})
            end
            obj.Listeners = [];
            obj.System_H.Remove_Link(obj.Name)
            error(me.message)
        end
    end

    %% link activation function
    % function that try to active link upon change. if not successful, it reverts the values back
    function try_to_link(obj,Source,Data,index)
        try
            obj.Link_Function(Source,Data,index) % the abstract function itself
            obj.Linked_Properties_Values_Revert = obj.Linked_Properties_Values; % if succeeded, updates the values revert list
        catch me % not succeeded
            obj.Linked_Properties_Values = obj.Linked_Properties_Values_Revert;
            for i = 1:length(obj.Linked_Properties_Values) % revert back the values  
                listeners_locations_array = obj.System_H.get_property_listeners_locations(obj.Linked_Properties{i});
                obj.System_H.turn_off_or_on_property_multi_listeners(listeners_locations_array,"off")
                obj.System_H.Change_Property_Link(obj.Linked_Properties{i},obj.Linked_Properties_Values(i))
                obj.System_H.turn_off_or_on_property_multi_listeners(listeners_locations_array,"on")
            end
            if obj.forGUI
                notify(obj,'ErrorInLink',EventDataClassCreation(me.message)) % will cause a UIAlert in the GUI
            else
                obj.System_H.Chain_Link_Error_Flag = 1; % flags link fail for possible chain links.
                error(me.message)
            end
            obj.Link_Fail_Flag = 1; % flags link fail.
            obj.System_H.Chain_Link_Error_Flag = 1; % flags link fail for possible chain links. (repeating it to suit both code and GUI scenarios)
        end
    end

    %% property meta data obtaining
    % a function that gets the component meta data, and the property's location in it.
    % used for validation of the properties for the link.
    % metadata -> the meta data of the component of the specific property
    % metaloc -> the location of the property in the meta data of the component
    % prop_i -> the index of the chosen property (in Linked_Properties) to validate

    function [metadata,metaloc] = get_property_meta(obj,prop_i)  

        % in case it is a component parameter with branch: (cases 5-6)
        if isnumeric(obj.Linked_Properties{prop_i}{1})  % examples: {1,'Duct1','Length'} or {1,'Duct1','Pressure',1}
            I = obj.System_H.findLoc(obj.Linked_Properties{prop_i}{2}); % finding location of the component in the system in order to reach it's handle
            metadata = metaclass(obj.System_H.Components_H{I}); % creating a meta class to the component in order to reach the parameter's info
            for j = 1:length(metadata.PropertyList) % searching for the location of the parameter
                if strcmp(metadata.PropertyList(j).Name,obj.Linked_Properties{prop_i}{3})
                    metaloc = j;
                    break
                end
            end
            % in case no metaloc was found, the parameter's name inserted was incorrect
            if ~exist("metaloc","var")
                error('Incorrect name of parameter was chosen')
            end

        else 
            % in case it is a component parameter without branch: (cases 3-4)
            if length(obj.Linked_Properties{prop_i}) >= 2 && ischar(obj.Linked_Properties{prop_i}{2})  % examples: {'Duct1','Length'} or {'Duct1','Pressure',1}
                I = obj.System_H.findLoc(obj.Linked_Properties{prop_i}{1}); % finding location of the component in the system in order to reach it's handle
                metadata = metaclass(obj.System_H.Components_H{I}); % creating a meta class to the component in order to reach the parameter's info
                for j = 1:length(metadata.PropertyList) % searching for the location of the parameter
                    if strcmp(metadata.PropertyList(j).Name,obj.Linked_Properties{prop_i}{2})
                        metaloc = j;
                        break
                    end
                end
                % in case no metaloc was found, the parameter's name inserted was incorrect
                if ~exist("metaloc","var")
                    error('Incorrect name of parameter was chosen')
                end

            % in case it is a system parameter: (cases 1-2)
            else   % examples: {'P_m'} or {'Begin',3}
                metadata = metaclass(obj.System_H); % creating a meta class to the system in order to reach the parameter's info
                for j = 1:length(metadata.PropertyList) % searching for the location of the parameter
                    if strcmp(metadata.PropertyList(j).Name,obj.Linked_Properties{prop_i}{1})
                        metaloc = j;
                        break
                    end
                end 
                % in case no metaloc was found, the parameter's name inserted was incorrect
                if ~exist("metaloc","var")
                    error('Incorrect name of parameter was chosen')
                end
            end 
        end
    end

    %% deleting link function %%% delete this function? i think there is no use to it
    function Delete_Link(obj,Delete_Components_Link) % not upadated to various parameters yet
        for i = 1:size(Delete_Components_Link,2)
            delete(obj.Listeners(Delete_Components_Link(i)))
            obj.Listeners(Delete_Components_Link(i)) = [];
       end
    end

    %% copy link
        function cp = Copy_Link(obj,varargin)
            %copy a link
            % Copy_Link() creates a copy of the link with the
            %                  name "copy of 'old name'"
            % Copy_Link(newName) creates a copy of the link with
            % the name specified
            
            % Shallow copy object
            cp = obj.copyElement;
            cp.System_H=TA_System.empty;
            cp.System=[];
            cp.Location=[];
            %cp.Empty_Varibles;
            
            if isempty(varargin)
                cp.Name=['copy of ',cp.Name];
            else
                cp.Name=varargin{1};
            end
        end
    %% rename function
    function rename(obj,value)           
        obj.Name=value;
        if ~isempty(obj.System)              
            obj.System_H.Links{obj.Location}=value;
        end
    end
end

methods (Abstract)
    Link_Function(obj,Source,Data,index)
    Specific_Link_Validation(obj)
end
events
    ErrorInLink
end
end

